"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeSavedObjectMigrationMaps = mergeSavedObjectMigrationMaps;
exports.mergeSavedObjectMigrations = mergeSavedObjectMigrations;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Composes two migrations into a single migration.
 * ```
 * mergeSavedObjectMigrations(outer, inner) -> (doc, context) => outer(inner(doc, context), context) }
 * ```
 *
 * If at least one of the migrations is not deferred, the composed migration will not be deferred.
 *
 * @public
 * @param outer Wrapping migration.
 * @param inner Wrapped migration.
 * @param rest Additional wrapped migrations to compose.
 * @returns The composed migration can be either a function or an object depending on the input migrations.
 */
function mergeSavedObjectMigrations(outer, inner, ...rest) {
  if (rest.length) {
    return mergeSavedObjectMigrations(mergeSavedObjectMigrations(outer, inner), ...rest);
  }
  if (!inner || !outer) {
    return inner || outer;
  }
  const innerMigration = (0, _lodash.isFunction)(inner) ? {
    transform: inner
  } : inner;
  const outerMigration = (0, _lodash.isFunction)(outer) ? {
    transform: outer
  } : outer;
  const merged = {
    deferred: !!(innerMigration.deferred && outerMigration.deferred),
    transform: (state, context) => outerMigration.transform(innerMigration.transform(state, context), context)
  };
  if ((0, _lodash.isFunction)(inner) && (0, _lodash.isFunction)(outer)) {
    return merged.transform;
  }
  return merged;
}

/**
 * Merges two saved object migration maps.
 *
 * If there is a migration for a given version on only one of the maps,
 * that migration function will be used:
 *
 * mergeSavedObjectMigrationMaps({ '1.2.3': f }, { '4.5.6': g }) -> { '1.2.3': f, '4.5.6': g }
 *
 * If there is a migration for a given version on both maps, the migrations will be composed:
 *
 * mergeSavedObjectMigrationMaps({ '1.2.3': f }, { '1.2.3': g }) -> { '1.2.3': (doc, context) => f(g(doc, context), context) }
 *
 * @public
 *
 * @param map1 - The first map to merge
 * @param map2 - The second map to merge
 * @returns The merged map {@link SavedObjectMigrationMap}
 */
function mergeSavedObjectMigrationMaps(map1, map2) {
  return (0, _lodash.mergeWith)({
    ...map1
  }, map2, (0, _lodash.ary)(mergeSavedObjectMigrations, 2));
}