"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerFavoritesUsageCollection = registerFavoritesUsageCollection;
var _favorites_saved_object = require("./favorites_saved_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function registerFavoritesUsageCollection({
  core,
  usageCollection
}) {
  usageCollection.registerCollector(usageCollection.makeUsageCollector({
    type: 'favorites',
    isReady: () => true,
    schema: {
      DYNAMIC_KEY /* e.g. 'dashboard' */: {
        total: {
          type: 'long',
          _meta: {
            description: 'Total favorite object count in this deployment'
          }
        },
        total_users_spaces: {
          type: 'long',
          _meta: {
            description: 'Total users per space that have favorited an object of this type in this deployment'
          }
        },
        avg_per_user_per_space: {
          type: 'double',
          _meta: {
            description: 'Average favorite objects count of this type per user per space for this deployment, only counts users who have favorited at least one object of this type'
          }
        },
        max_per_user_per_space: {
          type: 'long',
          _meta: {
            description: 'Max favorite objects count of this type per user per space for this deployment'
          }
        }
      }
    },
    fetch: async context => {
      var _response$aggregation, _response$aggregation2, _response$aggregation3;
      const favoritesIndex = await core.getStartServices().then(([{
        savedObjects
      }]) => savedObjects.getIndexForType(_favorites_saved_object.favoritesSavedObjectName));
      const response = await context.esClient.search({
        index: favoritesIndex,
        size: 0,
        _source: false,
        filter_path: ['aggregations'],
        query: {
          bool: {
            filter: [{
              term: {
                type: 'favorites'
              }
            }]
          }
        },
        runtime_mappings: {
          number_of_favorites: {
            type: 'long',
            script: {
              source: "emit(doc['favorites.favoriteIds'].length)"
            }
          }
        },
        aggs: {
          types: {
            terms: {
              field: 'favorites.type'
            },
            aggs: {
              stats: {
                stats: {
                  field: 'number_of_favorites'
                }
              }
            }
          }
        }
      });
      const favoritesUsage = {};
      const typesBuckets = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : (_response$aggregation3 = _response$aggregation2.types) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
      typesBuckets.forEach(bucket => {
        favoritesUsage[bucket.key] = {
          total: bucket.stats.sum,
          total_users_spaces: bucket.stats.count,
          avg_per_user_per_space: bucket.stats.avg,
          max_per_user_per_space: bucket.stats.max
        };
      });
      return favoritesUsage;
    }
  }));
}