"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getBenchmarksData = exports.getBenchmarks = void 0;
var _constants = require("../../../common/constants");
var _get_clusters = require("../compliance_dashboard/get_clusters");
var _get_stats = require("../compliance_dashboard/get_stats");
var _get_safe_posture_type_runtime_mapping = require("../../../common/runtime_mappings/get_safe_posture_type_runtime_mapping");
var _v = require("../benchmark_rules/get_states/v1");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getBenchmarksData = async (soClient, encryptedSoClient, esClient, logger) => {
  // Returns a list of benchmark based on their Version and Benchmark ID

  const benchmarksResponse = await soClient.find({
    type: _constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE,
    aggs: {
      benchmark_id: {
        terms: {
          field: `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.id`
        },
        aggs: {
          name: {
            terms: {
              field: `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.name`
            },
            aggs: {
              version: {
                terms: {
                  field: `${_constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE}.attributes.metadata.benchmark.version`
                }
              }
            }
          }
        }
      }
    },
    perPage: 0
  });
  const benchmarkAgg = benchmarksResponse.aggregations;
  const rulesFilter = await (0, _v.getMutedRulesFilterQuery)(encryptedSoClient);
  const {
    id: pitId
  } = await esClient.openPointInTime({
    index: _constants.LATEST_FINDINGS_INDEX_DEFAULT_NS,
    keep_alive: '30s'
  });
  // Transform response to a benchmark row: {id, name, version}
  // For each Benchmark entry : Calculate Score, Get amount of enrolled agents
  const result = await Promise.all(benchmarkAgg.benchmark_id.buckets.flatMap(async benchmark => {
    var _benchmark$name, _benchmark$name$bucke, _benchmark$name$bucke2;
    const benchmarkId = benchmark.key;
    const benchmarkName = benchmark.name.buckets[0].key;
    const benchmarksTableObjects = await Promise.all(benchmark === null || benchmark === void 0 ? void 0 : (_benchmark$name = benchmark.name) === null || _benchmark$name === void 0 ? void 0 : (_benchmark$name$bucke = _benchmark$name.buckets[0]) === null || _benchmark$name$bucke === void 0 ? void 0 : (_benchmark$name$bucke2 = _benchmark$name$bucke.version) === null || _benchmark$name$bucke2 === void 0 ? void 0 : _benchmark$name$bucke2.buckets.flatMap(async benchmarkObj => {
      const benchmarkVersion = benchmarkObj.key;
      const postureType = benchmarkId === 'cis_eks' || benchmarkId === 'cis_k8s' ? 'kspm' : 'cspm';
      const runtimeMappings = (0, _get_safe_posture_type_runtime_mapping.getSafePostureTypeRuntimeMapping)();
      const query = {
        bool: {
          filter: [{
            term: {
              'rule.benchmark.id': benchmarkId
            }
          }, {
            term: {
              'rule.benchmark.version': benchmarkVersion
            }
          }, {
            term: {
              safe_posture_type: postureType
            }
          }],
          must_not: rulesFilter
        }
      };
      const benchmarkScore = await (0, _get_stats.getStats)(esClient, query, pitId, runtimeMappings, logger);
      const benchmarkEvaluation = await (0, _get_clusters.getClusters)(esClient, query, pitId, runtimeMappings, logger);
      return {
        id: benchmarkId,
        name: benchmarkName,
        version: benchmarkVersion.replace('v', ''),
        score: benchmarkScore,
        evaluation: benchmarkEvaluation.length
      };
    }));
    return benchmarksTableObjects;
  }));
  return result.flat();
};
exports.getBenchmarksData = getBenchmarksData;
const getBenchmarks = async (esClient, soClient, encryptedSoClient, logger) => {
  const benchmarks = await getBenchmarksData(soClient, encryptedSoClient, esClient, logger);
  const getBenchmarkResponse = {
    items: benchmarks
  };
  return getBenchmarkResponse;
};
exports.getBenchmarks = getBenchmarks;