"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.healthRoute = void 0;
var _health = require("../../../../../common/routes/framework/apis/health");
var _lib = require("../../../lib");
var _types = require("../../../../types");
var _get_security_health = require("../../../../lib/get_security_health");
var _v = require("./transforms/transform_health_response/v1");
var _constants = require("../../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const healthRoute = (router, licenseState, encryptedSavedObjects) => {
  router.get({
    path: `${_types.BASE_ALERTING_API_PATH}/_health`,
    security: _constants.DEFAULT_ALERTING_ROUTE_SECURITY,
    options: {
      access: 'public',
      summary: `Get the alerting framework health`,
      tags: ['oas-tag:alerting']
    },
    validate: {
      request: {},
      response: {
        200: {
          body: () => _health.healthFrameworkResponseSchemaV1,
          description: 'Indicates a successful call.'
        },
        401: {
          description: 'Authorization information is missing or invalid.'
        }
      }
    }
  }, router.handleLegacyErrors((0, _lib.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    try {
      const alertingContext = await context.alerting;
      const rulesClient = await alertingContext.getRulesClient();
      // Verify that user has access to at least one rule type
      const ruleTypes = Array.from(await rulesClient.listRuleTypes());
      if (ruleTypes.length > 0) {
        const alertingFrameworkHealth = await alertingContext.getFrameworkHealth();
        const securityHealth = await (0, _get_security_health.getSecurityHealth)(async () => licenseState ? licenseState.getIsSecurityEnabled() : null, async () => encryptedSavedObjects.canEncrypt, alertingContext.areApiKeysEnabled);
        const frameworkHealth = {
          ...securityHealth,
          alertingFrameworkHealth
        };
        return res.ok({
          body: (0, _v.transformHealthBodyResponse)(frameworkHealth)
        });
      } else {
        return res.forbidden({
          body: {
            message: `Unauthorized to access alerting framework health`
          }
        });
      }
    } catch (error) {
      return res.badRequest({
        body: error
      });
    }
  })));
};
exports.healthRoute = healthRoute;