"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.softDeleteGaps = void 0;
var _alerting_event_logger = require("../../alerting_event_logger/alerting_event_logger");
var _process_all_rule_gaps = require("../process_all_rule_gaps");
var _soft_delete_gaps_batch = require("./soft_delete_gaps_batch");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Soft delete gaps for a given rule.
 * It orchestrates the process of searching and marking all the rule gaps as deleted
 */
const softDeleteGaps = async params => {
  const {
    ruleId,
    logger,
    eventLogClient,
    eventLogger
  } = params;
  try {
    if (!eventLogger) {
      throw new Error('Event logger is required');
    }
    const alertingEventLogger = new _alerting_event_logger.AlertingEventLogger(eventLogger);
    let hasErrors = false;
    const processGapsBatch = async fetchedGaps => {
      if (fetchedGaps.length > 0) {
        const success = await (0, _soft_delete_gaps_batch.softDeleteGapsBatch)({
          gaps: fetchedGaps,
          alertingEventLogger,
          logger,
          eventLogClient
        });
        if (!success) {
          hasErrors = true;
        }
      }
    };
    await (0, _process_all_rule_gaps.processAllRuleGaps)({
      ruleId,
      logger,
      eventLogClient,
      processGapsBatch,
      statuses: Object.values(_constants.gapStatus)
    });
    if (hasErrors) {
      throw new Error('Some gaps failed to soft delete');
    }
  } catch (e) {
    logger.error(`Failed to soft delete gaps for rule ${ruleId}: ${e.message}`);
  }
};
exports.softDeleteGaps = softDeleteGaps;