"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.significantItemsHandlerFactory = void 0;
var _async = require("async");
var _lodash = require("lodash");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _i18n = require("@kbn/i18n");
var _stream_reducer = require("@kbn/aiops-log-rate-analysis/api/stream_reducer");
var _queue_field_candidates = require("@kbn/aiops-log-rate-analysis/queue_field_candidates");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _fetch_significant_categories = require("@kbn/aiops-log-rate-analysis/queries/fetch_significant_categories");
var _fetch_significant_term_p_values = require("@kbn/aiops-log-rate-analysis/queries/fetch_significant_term_p_values");
var _constants = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const significantItemsHandlerFactory = ({
  abortSignal,
  esClient,
  logDebugMessage,
  logger,
  requestBody,
  responseStream,
  stateHandler,
  version
}) => async ({
  keywordFieldCandidates,
  textFieldCandidates
}) => {
  var _overrides$significan, _overrides, _overrides$significan2, _overrides$significan3, _overrides2, _overrides2$significa, _requestBody$override;
  // This will store the combined count of detected significant log patterns and keywords
  let fieldValuePairsCount = 0;
  const significantCategories = [];
  significantCategories.push(...((_overrides$significan = (_overrides = requestBody.overrides) === null || _overrides === void 0 ? void 0 : (_overrides$significan2 = _overrides.significantItems) === null || _overrides$significan2 === void 0 ? void 0 : _overrides$significan2.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.LOG_PATTERN)) !== null && _overrides$significan !== void 0 ? _overrides$significan : []));
  const significantTerms = [];
  significantTerms.push(...((_overrides$significan3 = (_overrides2 = requestBody.overrides) === null || _overrides2 === void 0 ? void 0 : (_overrides2$significa = _overrides2.significantItems) === null || _overrides2$significa === void 0 ? void 0 : _overrides2$significa.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.KEYWORD)) !== null && _overrides$significan3 !== void 0 ? _overrides$significan3 : []));
  let remainingKeywordFieldCandidates;
  let remainingTextFieldCandidates;
  let loadingStepSizePValues;
  if ((_requestBody$override = requestBody.overrides) !== null && _requestBody$override !== void 0 && _requestBody$override.loaded) {
    var _requestBody$override2;
    loadingStepSizePValues = _constants.LOADED_FIELD_CANDIDATES + _constants.PROGRESS_STEP_P_VALUES - ((_requestBody$override2 = requestBody.overrides) === null || _requestBody$override2 === void 0 ? void 0 : _requestBody$override2.loaded);
  } else {
    loadingStepSizePValues = _constants.PROGRESS_STEP_P_VALUES;
  }
  if (version === '2') {
    var _overrides3;
    const overridesRemainingFieldCandidates = (_overrides3 = requestBody.overrides) === null || _overrides3 === void 0 ? void 0 : _overrides3.remainingFieldCandidates;
    if (Array.isArray(overridesRemainingFieldCandidates)) {
      keywordFieldCandidates.push(...overridesRemainingFieldCandidates);
      remainingKeywordFieldCandidates = overridesRemainingFieldCandidates;
    } else {
      remainingKeywordFieldCandidates = keywordFieldCandidates;
    }
    remainingTextFieldCandidates = textFieldCandidates;
  } else if (version === '3') {
    var _overrides4, _overrides5;
    const overridesRemainingKeywordFieldCandidates = (_overrides4 = requestBody.overrides) === null || _overrides4 === void 0 ? void 0 : _overrides4.remainingKeywordFieldCandidates;
    if (Array.isArray(overridesRemainingKeywordFieldCandidates)) {
      keywordFieldCandidates.push(...overridesRemainingKeywordFieldCandidates);
      remainingKeywordFieldCandidates = overridesRemainingKeywordFieldCandidates;
    } else {
      remainingKeywordFieldCandidates = keywordFieldCandidates;
    }
    const overridesRemainingTextFieldCandidates = (_overrides5 = requestBody.overrides) === null || _overrides5 === void 0 ? void 0 : _overrides5.remainingTextFieldCandidates;
    if (Array.isArray(overridesRemainingTextFieldCandidates)) {
      textFieldCandidates.push(...overridesRemainingTextFieldCandidates);
      remainingTextFieldCandidates = overridesRemainingTextFieldCandidates;
    } else {
      remainingTextFieldCandidates = textFieldCandidates;
    }
  }
  logDebugMessage('Fetch p-values.');
  const pValuesQueueChunks = [...(0, _lodash.chunk)(textFieldCandidates, _queue_field_candidates.QUEUE_CHUNKING_SIZE).map(d => ({
    textFieldCandidates: d
  })), ...(0, _lodash.chunk)(keywordFieldCandidates, _queue_field_candidates.QUEUE_CHUNKING_SIZE).map(d => ({
    keywordFieldCandidates: d
  }))];
  const loadingStepSize = 1 / pValuesQueueChunks.length * loadingStepSizePValues;
  const pValuesQueue = (0, _async.queue)(async function (payload) {
    if ((0, _queue_field_candidates.isKeywordFieldCandidates)(payload)) {
      const {
        keywordFieldCandidates: fieldNames
      } = payload;
      let pValues;
      try {
        pValues = await (0, _fetch_significant_term_p_values.fetchSignificantTermPValues)({
          esClient,
          abortSignal,
          logger,
          emitError: responseStream.pushError,
          arguments: {
            ...requestBody,
            fieldNames,
            sampleProbability: stateHandler.sampleProbability()
          }
        });
      } catch (e) {
        if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
          logger.error(`Failed to fetch p-values for ${fieldNames.join()}, got: \n${e.toString()}`);
          responseStream.pushError(`Failed to fetch p-values for ${fieldNames.join()}.`);
        }
        return;
      }
      remainingKeywordFieldCandidates = remainingKeywordFieldCandidates.filter(d => !fieldNames.includes(d));
      if (pValues.length > 0) {
        significantTerms.push(...pValues);
        responseStream.push((0, _stream_reducer.addSignificantItems)(pValues));
        fieldValuePairsCount += pValues.length;
      }
    } else if ((0, _queue_field_candidates.isTextFieldCandidates)(payload)) {
      const {
        textFieldCandidates: fieldNames
      } = payload;
      let significantCategoriesForField;
      try {
        significantCategoriesForField = await (0, _fetch_significant_categories.fetchSignificantCategories)({
          esClient,
          logger,
          emitError: responseStream.pushError,
          abortSignal,
          arguments: {
            ...requestBody,
            fieldNames,
            sampleProbability: stateHandler.sampleProbability()
          }
        });
      } catch (e) {
        if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
          logger.error(`Failed to fetch p-values for ${fieldNames.join()}, got: \n${e.toString()}`);
          responseStream.pushError(`Failed to fetch p-values for ${fieldNames.join()}.`);
        }
        return;
      }
      remainingTextFieldCandidates = remainingTextFieldCandidates.filter(d => !fieldNames.includes(d));
      if (significantCategoriesForField.length > 0) {
        significantCategories.push(...significantCategoriesForField);
        responseStream.push((0, _stream_reducer.addSignificantItems)(significantCategoriesForField));
        fieldValuePairsCount += significantCategoriesForField.length;
      }
    }
    stateHandler.loaded(loadingStepSize, false);
    responseStream.push((0, _stream_reducer.updateLoadingState)({
      ccsWarning: false,
      loaded: stateHandler.loaded(),
      loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.identifiedFieldValuePairs', {
        defaultMessage: 'Identified {fieldValuePairsCount, plural, one {# significant field/value pair} other {# significant field/value pairs}}.',
        values: {
          fieldValuePairsCount
        }
      }),
      remainingKeywordFieldCandidates,
      remainingTextFieldCandidates
    }));
  }, _constants.MAX_CONCURRENT_QUERIES);

  // This chunks keyword and text field candidates, then passes them on
  // to the async queue for processing. Each chunk will be part of a single
  // query using multiple aggs for each candidate. For many candidates,
  // on top of that the async queue will process multiple queries concurrently.
  pValuesQueue.push(pValuesQueueChunks, err => {
    if (err) {
      logger.error(`Failed to fetch p-values.', got: \n${err.toString()}`);
      responseStream.pushError(`Failed to fetch p-values.`);
      pValuesQueue.kill();
      responseStream.end();
    } else if (stateHandler.shouldStop()) {
      logDebugMessage('shouldStop fetching p-values.');
      pValuesQueue.kill();
      responseStream.end();
    }
  });
  await pValuesQueue.drain();
  fieldValuePairsCount = significantCategories.length + significantTerms.length;
  if (fieldValuePairsCount === 0) {
    logDebugMessage('Stopping analysis, did not find significant terms.');
    responseStream.endWithUpdatedLoadingState();
    return;
  }
  return {
    fieldValuePairsCount,
    significantCategories,
    significantTerms
  };
};
exports.significantItemsHandlerFactory = significantItemsHandlerFactory;