"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCategoryCountRequest = exports.getCategoryCountMSearchRequest = exports.fetchCategoryCounts = void 0;
var _lodash = require("lodash");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _get_category_query = require("@kbn/aiops-log-pattern-analysis/get_category_query");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _get_query_with_params = require("./get_query_with_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isMsearchResponseItem = arg => (0, _mlIsPopulatedObject.isPopulatedObject)(arg, ['hits']);
const getCategoryCountRequest = (params, fieldName, category, from, to) => {
  const {
    index
  } = params;
  const categoryQuery = (0, _get_category_query.getCategoryQuery)(fieldName, [category]);
  const query = (0, _get_query_with_params.getQueryWithParams)({
    // This will override the original start/end params if
    // the optional from/to args are provided.
    params: {
      ...params,
      ...(from ? {
        start: from
      } : {}),
      ...(to ? {
        end: to
      } : {})
    },
    filter: categoryQuery
  });
  return {
    index,
    body: {
      query,
      size: 0,
      track_total_hits: true
    }
  };
};
exports.getCategoryCountRequest = getCategoryCountRequest;
const getCategoryCountMSearchRequest = (params, fieldName, categories, from, to) => categories.flatMap(category => [{
  index: params.index
}, getCategoryCountRequest(params, fieldName, category, from, to).body]);
exports.getCategoryCountMSearchRequest = getCategoryCountMSearchRequest;
const fetchCategoryCounts = async (esClient, params, fieldName, categories, from, to, logger, emitError, abortSignal) => {
  const updatedCategories = (0, _lodash.cloneDeep)(categories);
  const searches = getCategoryCountMSearchRequest(params, fieldName, categories.categories, from, to);
  let mSearchresponse;
  try {
    mSearchresponse = await esClient.msearch({
      searches
    }, {
      signal: abortSignal,
      maxRetries: 0
    });
  } catch (error) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(error)) {
      if (logger) {
        logger.error(`Failed to fetch category counts for field name "${fieldName}", got: \n${JSON.stringify(error, null, 2)}`);
      }
      if (emitError) {
        emitError(`Failed to fetch category counts for field name "${fieldName}".`);
      }
    }
    return updatedCategories;
  }
  for (const [index, resp] of mSearchresponse.responses.entries()) {
    if (isMsearchResponseItem(resp)) {
      var _value;
      updatedCategories.categories[index].count = (_value = resp.hits.total.value) !== null && _value !== void 0 ? _value : 0;
    } else {
      if (logger) {
        logger.error(`Failed to fetch category count for category "${updatedCategories.categories[index].key}", got: \n${JSON.stringify(resp, null, 2)}`);
      }
      if (emitError) {
        emitError(`Failed to fetch category count for category "${updatedCategories.categories[index].key}".`);
      }
    }
  }
  return updatedCategories;
};
exports.fetchCategoryCounts = fetchCategoryCounts;