"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VegaBaseView = void 0;
exports.bypassExternalUrlCheck = bypassExternalUrlCheck;
var _moment = _interopRequireDefault(require("moment"));
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _vega = require("vega");
var _vegaInterpreter = require("vega-interpreter");
var _vegaLite = require("vega-lite");
var _utils = require("../data_model/utils");
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _vega_tooltip = require("./vega_tooltip");
var _services = require("../services");
var _extract_index_pattern = require("../lib/extract_index_pattern");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Vega's extension functions are global. When called,
// we forward execution to the instance-specific handler
// This functions must be declared in the VegaBaseView class
const vegaFunctions = {
  kibanaAddFilter: 'addFilterHandler',
  kibanaRemoveFilter: 'removeFilterHandler',
  kibanaRemoveAllFilters: 'removeAllFiltersHandler',
  kibanaSetTimeFilter: 'setTimeFilterHandler'
};
for (const funcName of Object.keys(vegaFunctions)) {
  if (!(0, _vega.expressionFunction)(funcName)) {
    (0, _vega.expressionFunction)(funcName, function handlerFwd(...args) {
      const view = this.context.dataflow;
      view.runAfter(() => view._kibanaView.vegaFunctionsHandler(funcName, ...args));
    });
  }
}
const bypassToken = Symbol();
function bypassExternalUrlCheck(url) {
  // processed in the  loader.sanitize  below
  return {
    url,
    bypassToken
  };
}
const getExternalUrlsAreNotEnabledError = () => new Error(_i18n.i18n.translate('visTypeVega.vegaParser.baseView.externalUrlsAreNotEnabledErrorMessage', {
  defaultMessage: 'External URLs are not enabled. Add {enableExternalUrls} to {kibanaConfigFileName}',
  values: {
    enableExternalUrls: 'vis_type_vega.enableExternalUrls: true',
    kibanaConfigFileName: 'kibana.yml'
  }
}));
const getExternalUrlServiceError = uri => new Error(_i18n.i18n.translate('visTypeVega.vegaParser.baseView.externalUrlServiceErrorMessage', {
  defaultMessage: 'External URL [{uri}] was denied by ExternalUrl service. You can configure external URL policies using "{externalUrlPolicy}" setting in {kibanaConfigFileName}.',
  values: {
    uri,
    externalUrlPolicy: 'externalUrl.policy',
    kibanaConfigFileName: 'kibana.yml'
  }
}));
class VegaBaseView {
  constructor(opts) {
    this._parentEl = opts.parentEl;
    this._parser = opts.vegaParser;
    this._serviceSettings = opts.serviceSettings;
    this._filterManager = opts.filterManager;
    this._fireEvent = opts.fireEvent;
    this._timefilter = opts.timefilter;
    this._view = null;
    this._vegaViewConfig = null;
    this._messages = null;
    this._destroyHandlers = [];
    this._initialized = false;
    this._externalUrl = opts.externalUrl;
    this._enableExternalUrls = (0, _services.getEnableExternalUrls)();
    this._renderMode = opts.renderMode;
    this._vegaStateRestorer = opts.vegaStateRestorer;
  }
  async init() {
    if (this._initialized) throw new Error(); // safety
    this._initialized = true;
    try {
      if (this._parser.useResize) {
        this._parentEl.classList.add('vgaVis--autoresize');
      } else {
        this._parentEl.classList.remove('vgaVis--autoresize');
      }
      this._parentEl.replaceChildren();
      this._parentEl.classList.add('vgaVis');
      this._parentEl.style.flexDirection = this._parser.containerDir;

      // bypass the onWarn warning checks - in some cases warnings may still need to be shown despite being disabled
      for (const warn of this._parser.warnings) {
        this._addMessage('warn', warn);
      }
      if (this._parser.error) {
        this.onError(this._parser.error);
        return;
      }
      this._container = document.createElement('div');
      this._container.classList.add('vgaVis__view');
      this._parentEl.append(this._container);
      this._controls = document.createElement('div');
      this._controls.classList.add(`vgaVis__controls`, `vgaVis__controls--${this._parser.controlsDir}`);
      this._parentEl.append(this._controls);
      this._addDestroyHandler(() => {
        if (this._container) {
          this._container.remove();
          this._container = null;
        }
        if (this._controls) {
          this._controls.remove();
          this._controls = null;
        }
        if (this._messages) {
          this._messages.remove();
          this._messages = null;
        }
        if (this._view) {
          const state = this._view.getState();
          if (state) {
            this._vegaStateRestorer.save(state);
          }
          this._view.finalize();
        }
        this._view = null;
        this._vegaViewConfig = null;
      });
      this._vegaViewConfig = this.createViewConfig();

      // The derived class should create this method
      await this._initViewCustomizations();
    } catch (err) {
      this.onError(err);
    }
  }

  /**
   * Find index pattern by its title, if not given, gets it from spec or a defaults one
   * @param {string} [index]
   * @returns {Promise<string>} index id
   */
  async findIndex(index) {
    const dataViews = (0, _services.getDataViews)();
    let idxObj;
    if (index) {
      [idxObj] = await dataViews.find(index, 1);
      if (!idxObj) {
        throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.baseView.indexNotFoundErrorMessage', {
          defaultMessage: 'Index {index} not found',
          values: {
            index: `"${index}"`
          }
        }));
      }
    } else {
      [idxObj] = await (0, _extract_index_pattern.extractIndexPatternsFromSpec)(this._parser.isVegaLite ? this._parser.vlspec : this._parser.spec);
      if (!idxObj) {
        const defaultIdx = await dataViews.getDefault();
        if (defaultIdx) {
          idxObj = defaultIdx;
        } else {
          throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.baseView.unableToFindDefaultIndexErrorMessage', {
            defaultMessage: 'Unable to find default index'
          }));
        }
      }
    }
    return idxObj.id;
  }
  handleExternalUrlError(externalUrlError) {
    this.onError(externalUrlError);
    throw externalUrlError;
  }
  createViewConfig() {
    var _vegaSpec$usermeta, _vegaSpec$usermeta$em;
    const config = {
      expr: _vegaInterpreter.expressionInterpreter,
      renderer: this._parser.renderer
    };

    // Override URL sanitizer to prevent external data loading (if disabled)
    const vegaLoader = (0, _vega.loader)();
    const originalSanitize = vegaLoader.sanitize.bind(vegaLoader);
    vegaLoader.sanitize = async (uri, options) => {
      if (uri.bypassToken === bypassToken) {
        // If uri has a bypass token, the uri was encoded by bypassExternalUrlCheck() above.
        // because user can only supply pure JSON data structure.
        uri = uri.url;
      } else if (!this._externalUrl.isInternalUrl(uri)) {
        if (!this._enableExternalUrls) {
          this.handleExternalUrlError(getExternalUrlsAreNotEnabledError());
        } else if (!this._externalUrl.validateUrl(uri)) {
          this.handleExternalUrlError(getExternalUrlServiceError(uri));
        }
      }
      const result = await originalSanitize(uri, options);
      // This will allow Vega users to load images from any domain.
      result.crossOrigin = null;
      return result;
    };
    const vegaSpec = this._parser.isVegaLite ? this._parser.vlspec : this._parser.spec;
    const usermetaLoaderOptions = (_vegaSpec$usermeta = vegaSpec.usermeta) === null || _vegaSpec$usermeta === void 0 ? void 0 : (_vegaSpec$usermeta$em = _vegaSpec$usermeta.embedOptions) === null || _vegaSpec$usermeta$em === void 0 ? void 0 : _vegaSpec$usermeta$em.loader;
    vegaLoader.options = usermetaLoaderOptions !== null && usermetaLoaderOptions !== void 0 ? usermetaLoaderOptions : {};
    config.loader = vegaLoader;
    const vegaLogger = (0, _vega.logger)(_vega.Warn);
    vegaLogger.warn = this.onWarn.bind(this);
    vegaLogger.error = this.onError.bind(this);
    config.logger = vegaLogger;
    return config;
  }
  onError() {
    var _this$_parser$searchA;
    const error = _utils.Utils.formatErrorToStr(...arguments);
    this._addMessage('err', error);
    (_this$_parser$searchA = this._parser.searchAPI.inspectorAdapters) === null || _this$_parser$searchA === void 0 ? void 0 : _this$_parser$searchA.vega.setError(error);
  }
  onWarn() {
    if (this._renderMode !== 'view' && (!this._parser || !this._parser.hideWarnings)) {
      this._addMessage('warn', _utils.Utils.formatWarningToStr(...arguments));
    }
  }
  _addMessage(type, text) {
    if (!this._messages) {
      this._messages = document.createElement('ul');
      this._messages.classList.add('vgaVis__messages');
      this._parentEl.append(this._messages);
    }
    const isMessageAlreadyDisplayed = [...this._messages.querySelectorAll(`:scope pre.vgaVis__messageCode`)].filter((index, element) => element.textContent === text).length;
    if (!isMessageAlreadyDisplayed) {
      const messageCodeEl = document.createElement('pre');
      messageCodeEl.classList.add('vgaVis__messageCode');
      messageCodeEl.textContent = text;
      const messageItemEl = document.createElement('li');
      messageItemEl.classList.add(`vgaVis__message`, `vgaVis__message--${type}`);
      messageItemEl.append(messageCodeEl);
      this._messages.append(messageItemEl);
    }
  }
  async resize(dimensions) {
    if (this._parser.useResize && this._view) {
      var _this$onViewContainer;
      this.updateVegaSize(this._view, dimensions);
      await this._view.runAsync();

      // The derived class should create this method
      (_this$onViewContainer = this.onViewContainerResize) === null || _this$onViewContainer === void 0 ? void 0 : _this$onViewContainer.call(this);
    }
  }
  updateVegaSize(view, dimensions) {
    var _dimensions$width, _dimensions$height;
    const width = Math.floor(Math.max(0, (_dimensions$width = dimensions === null || dimensions === void 0 ? void 0 : dimensions.width) !== null && _dimensions$width !== void 0 ? _dimensions$width : this._container.clientWidth - 1));
    const height = Math.floor(Math.max(0, (_dimensions$height = dimensions === null || dimensions === void 0 ? void 0 : dimensions.height) !== null && _dimensions$height !== void 0 ? _dimensions$height : this._container.clientHeight - 1));
    if (view.width() !== width || view.height() !== height) {
      view.width(width).height(height);
      return true;
    }
    return false;
  }
  setView(view) {
    if (this._view === view) return;
    if (this._view) {
      this._view.finalize();
    }
    this._view = view;
    if (view) {
      // Global vega expression handler uses it to call custom functions
      view._kibanaView = this;
      if (this._parser.tooltips) {
        // position and padding can be specified with
        // {config:{kibana:{tooltips: {position: 'top', padding: 15 } }}}
        const tthandler = new _vega_tooltip.TooltipHandler(this._container, view, this._parser.tooltips);

        // Vega bug workaround - need to destroy tooltip by hand
        this._addDestroyHandler(() => tthandler.hideTooltip());
      }
      const state = this._vegaStateRestorer.restore();
      if (state) {
        return view.setState(state);
      } else {
        return view.runAsync();
      }
    }
  }

  /**
   * Handle
   * @param funcName
   * @param args
   * @returns {Promise<void>}
   */
  async vegaFunctionsHandler(funcName, ...args) {
    try {
      const handlerFunc = vegaFunctions[funcName];
      if (!handlerFunc || !this[handlerFunc]) {
        // in case functions don't match the list above
        throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.baseView.functionIsNotDefinedForGraphErrorMessage', {
          defaultMessage: '{funcName} is not defined for this graph',
          values: {
            funcName: `${funcName}()`
          }
        }));
      }
      await this[handlerFunc](...args);
    } catch (err) {
      this.onError(err);
    }
  }

  /**
   * @param {object} query Elastic Query DSL snippet, as used in the query DSL editor
   * @param {string} [index] as defined in Kibana, or default if missing
   * @param {string} Elastic Query DSL's Custom label for kibanaAddFilter, as used in '+ Add Filter'
   */
  async addFilterHandler(query, index, alias) {
    const normalizedQuery = (0, _utils2.normalizeObject)(query);
    const normalizedIndex = (0, _utils2.normalizeString)(index);
    const normalizedAlias = (0, _utils2.normalizeString)(alias);
    const indexId = await this.findIndex(normalizedIndex);
    const filter = (0, _esQuery.buildQueryFilter)(normalizedQuery, indexId, normalizedAlias);
    this._fireEvent({
      name: 'applyFilter',
      data: {
        filters: [filter]
      }
    });
  }

  /**
   * @param {object} query Elastic Query DSL snippet, as used in the query DSL editor
   * @param {string} [index] as defined in Kibana, or default if missing
   */
  async removeFilterHandler(query, index) {
    const normalizedQuery = (0, _utils2.normalizeObject)(query);
    const normalizedIndex = (0, _utils2.normalizeString)(index);
    const indexId = await this.findIndex(normalizedIndex);
    const filterToRemove = (0, _esQuery.buildQueryFilter)(normalizedQuery, indexId);
    const currentFilters = this._filterManager.getFilters();
    const existingFilter = currentFilters.find(filter => (0, _esQuery.compareFilters)(filter, filterToRemove));
    if (!existingFilter) return;
    try {
      this._filterManager.removeFilter(existingFilter);
    } catch (err) {
      this.onError(err);
    }
  }
  removeAllFiltersHandler() {
    this._filterManager.removeAll();
  }

  /**
   * Update dashboard time filter to the new values
   * @param {number|string|Date} start
   * @param {number|string|Date} end
   */
  setTimeFilterHandler(start, end) {
    const normalizedStart = (0, _utils2.normalizeDate)(start);
    const normalizedEnd = (0, _utils2.normalizeDate)(end);
    const {
      from,
      to,
      mode
    } = VegaBaseView._parseTimeRange(normalizedStart, normalizedEnd);
    this._fireEvent({
      name: 'applyFilter',
      data: {
        timeFieldName: '*',
        filters: [{
          query: {
            range: {
              '*': {
                mode,
                gte: from,
                lte: to
              }
            }
          }
        }]
      }
    });
  }

  /**
   * Parse start and end values, determining the mode, and if order should be reversed
   * @internal
   */
  static _parseTimeRange(start, end) {
    const absStart = (0, _moment.default)(start);
    const absEnd = (0, _moment.default)(end);
    const isValidAbsStart = absStart.isValid();
    const isValidAbsEnd = absEnd.isValid();
    let mode = 'absolute';
    let from;
    let to;
    let reverse;
    if (isValidAbsStart && isValidAbsEnd) {
      // Both are valid absolute dates.
      from = absStart;
      to = absEnd;
      reverse = absStart.isAfter(absEnd);
    } else {
      // Try to parse as relative dates too (absolute dates will also be accepted)
      const startDate = _datemath.default.parse(start);
      const endDate = _datemath.default.parse(end);
      if (!startDate || !endDate || !startDate.isValid() || !endDate.isValid()) {
        throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.baseView.timeValuesTypeErrorMessage', {
          defaultMessage: 'Error setting time filter: both time values must be either relative or absolute dates. {start}, {end}',
          values: {
            start: `start=${JSON.stringify(start)}`,
            end: `end=${JSON.stringify(end)}`
          }
        }));
      }
      reverse = startDate.isAfter(endDate);
      if (isValidAbsStart || isValidAbsEnd) {
        // Mixing relative and absolute - treat them as absolute
        from = startDate;
        to = endDate;
      } else {
        // Both dates are relative
        mode = 'relative';
        from = start;
        to = end;
      }
    }
    if (reverse) {
      [from, to] = [to, from];
    }
    return {
      from,
      to,
      mode
    };
  }

  /**
   * Set global debug variable to simplify vega debugging in console. Show info message first time
   */
  setDebugValues(view, spec, vlspec) {
    var _this$_parser$searchA2;
    (_this$_parser$searchA2 = this._parser.searchAPI.inspectorAdapters) === null || _this$_parser$searchA2 === void 0 ? void 0 : _this$_parser$searchA2.vega.bindInspectValues({
      view,
      spec: vlspec || spec
    });
    if (window) {
      if (window.VEGA_DEBUG === undefined && console) {
        console.log('%cWelcome to Kibana Vega Plugin!', 'font-size: 16px; font-weight: bold;');
        console.log('You can access the Vega view with VEGA_DEBUG. ' + 'Learn more at https://vega.github.io/vega/docs/api/debugging/.');
      }
      const debugObj = {};
      window.VEGA_DEBUG = debugObj;
      window.VEGA_DEBUG.VEGA_VERSION = _vega.version;
      window.VEGA_DEBUG.VEGA_LITE_VERSION = _vegaLite.version;
      window.VEGA_DEBUG.view = view;
      window.VEGA_DEBUG.vega_spec = spec;
      window.VEGA_DEBUG.vegalite_spec = vlspec;

      // On dispose, clean up, but don't use undefined to prevent repeated debug statements
      this._addDestroyHandler(() => {
        if (debugObj === window.VEGA_DEBUG) {
          window.VEGA_DEBUG.view = null;
          window.VEGA_DEBUG.vega_spec = null;
          window.VEGA_DEBUG.vegalite_spec = null;
          window.VEGA_DEBUG = null;
        }
      });
    }
  }
  destroy() {
    // properly handle multiple destroy() calls by converting this._destroyHandlers
    // into the _ongoingDestroy promise, while handlers are being disposed
    if (this._destroyHandlers) {
      // If no destroy is yet running, execute all handlers and wait for all of them to resolve.
      this._ongoingDestroy = Promise.all(this._destroyHandlers.map(v => v()));
      this._destroyHandlers = null;
    }
    return this._ongoingDestroy;
  }
  _addDestroyHandler(handler) {
    // If disposing hasn't started yet, enqueue it, otherwise dispose right away
    // This creates a minor issue - if disposing has started but not yet finished,
    // and we dispose the new handler right away, the destroy() does not wait for it.
    // This behavior is no different from the case when disposing has already completed,
    // so it shouldn't create any issues.
    if (this._destroyHandlers) {
      this._destroyHandlers.push(handler);
    } else {
      handler();
    }
  }
}
exports.VegaBaseView = VegaBaseView;