"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerReindexIndicesRoutes = registerReindexIndicesRoutes;
var _configSchema = require("@kbn/config-schema");
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../../../common/constants");
var _types = require("../../../common/types");
var _es_version_precheck = require("../../lib/es_version_precheck");
var _reindexing = require("../../lib/reindexing");
var _reindex_actions = require("../../lib/reindexing/reindex_actions");
var _map_any_error_to_kibana_http_response = require("./map_any_error_to_kibana_http_response");
var _reindex_handler = require("./reindex_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerReindexIndicesRoutes({
  credentialStore,
  router,
  licensing,
  log,
  getSecurityPlugin,
  lib: {
    handleEsError
  }
}, getWorker) {
  const BASE_PATH = `${_constants.API_BASE_PATH}/reindex`;

  // Start reindex for an index
  router.post({
    path: `${BASE_PATH}/{indexName}`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es and saved object clients for authorization'
      }
    },
    options: {
      access: 'public',
      summary: `Start or resume reindex`
    },
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, (0, _es_version_precheck.versionCheckHandlerWrapper)(async ({
    core
  }, request, response) => {
    const {
      savedObjects: {
        getClient
      },
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      indexName
    } = request.params;
    try {
      const result = await (0, _reindex_handler.reindexHandler)({
        savedObjects: getClient({
          includedHiddenTypes: [_types.REINDEX_OP_TYPE]
        }),
        dataClient: esClient,
        indexName,
        log,
        licensing,
        request,
        credentialStore,
        security: getSecurityPlugin()
      });

      // Kick the worker on this node to immediately pickup the new reindex operation.
      getWorker().forceRefresh();
      return response.ok({
        body: result
      });
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(error);
    }
  }));

  // Get status
  router.get({
    path: `${BASE_PATH}/{indexName}`,
    options: {
      access: 'public',
      summary: `Get reindex status`
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es and saved object clients for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, (0, _es_version_precheck.versionCheckHandlerWrapper)(async ({
    core
  }, request, response) => {
    const {
      savedObjects,
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      getClient
    } = savedObjects;
    const {
      indexName
    } = request.params;
    const asCurrentUser = esClient.asCurrentUser;
    const reindexActions = (0, _reindex_actions.reindexActionsFactory)(getClient({
      includedHiddenTypes: [_types.REINDEX_OP_TYPE]
    }), asCurrentUser, log);
    const reindexService = (0, _reindexing.reindexServiceFactory)(asCurrentUser, reindexActions, log, licensing);
    try {
      const hasRequiredPrivileges = await reindexService.hasRequiredPrivileges(indexName);
      const reindexOp = await reindexService.findReindexOperation(indexName);
      // If the user doesn't have privileges than querying for warnings is going to fail.
      const warnings = hasRequiredPrivileges ? await reindexService.detectReindexWarnings(indexName) : [];
      const isTruthy = value => value === true || value === 'true';
      const {
        aliases,
        settings,
        isInDataStream,
        isFollowerIndex
      } = await reindexService.getIndexInfo(indexName);
      const body = {
        reindexOp: reindexOp ? reindexOp.attributes : undefined,
        warnings,
        hasRequiredPrivileges,
        meta: {
          indexName,
          reindexName: (0, _reindexing.generateNewIndexName)(indexName),
          aliases: Object.keys(aliases),
          isFrozen: isTruthy(settings === null || settings === void 0 ? void 0 : settings.frozen),
          isReadonly: isTruthy(settings === null || settings === void 0 ? void 0 : settings.verified_read_only),
          isInDataStream,
          isFollowerIndex
        }
      };
      return response.ok({
        body
      });
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(error);
    }
  }));

  // Cancel reindex
  router.post({
    path: `${BASE_PATH}/{indexName}/cancel`,
    options: {
      access: 'public',
      summary: `Cancel reindex`
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es and saved object clients for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, (0, _es_version_precheck.versionCheckHandlerWrapper)(async ({
    core
  }, request, response) => {
    const {
      savedObjects,
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      indexName
    } = request.params;
    const {
      getClient
    } = savedObjects;
    const callAsCurrentUser = esClient.asCurrentUser;
    const reindexActions = (0, _reindex_actions.reindexActionsFactory)(getClient({
      includedHiddenTypes: [_types.REINDEX_OP_TYPE]
    }), callAsCurrentUser, log);
    const reindexService = (0, _reindexing.reindexServiceFactory)(callAsCurrentUser, reindexActions, log, licensing);
    try {
      await reindexService.cancelReindexing(indexName);
      return response.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(error);
    }
  }));
}