"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sourceNameForIndex = exports.getReindexWarnings = exports.getDeprecatedSettingWarning = exports.generateNewIndexName = void 0;
var _version = require("../version");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * An array of deprecated index settings specific to 7.0 --> 8.0 upgrade
 * This excludes the deprecated translog retention settings
 * as these are only marked as deprecated if soft deletes is enabled
 * See logic in getDeprecatedSettingWarning() for more details
 */
const deprecatedSettings = ['index.force_memory_term_dictionary', 'index.max_adjacency_matrix_filters', 'index.soft_deletes.enabled'];

/**
 * Provides the assumed source of the index name stripping any prefixing
 * introduced by the upgrade assistant
 *
 * Examples:
 *   .reindex-v7-foo => .foo
 *   reindex-v7-foo => foo
 *
 * @param indexName
 */
const sourceNameForIndex = indexName => {
  const matches = indexName.match(/^([\.])?(.*)$/) || [];
  const internal = matches[1] || '';
  const baseName = matches[2];

  // in 6.7+ we prepend to avoid conflicts with index patterns/templates/etc
  const reindexedMatcher = new RegExp(`reindexed-v${_version.versionService.getPrevMajorVersion()}-`, 'g');
  const cleanBaseName = baseName.replace(reindexedMatcher, '');
  return `${internal}${cleanBaseName}`;
};

/**
 * Provides the index name to re-index into
 *
 * .foo -> .reindexed-v7-foo
 * foo => reindexed-v7-foo
 */
exports.sourceNameForIndex = sourceNameForIndex;
const generateNewIndexName = indexName => {
  const sourceName = sourceNameForIndex(indexName);
  const currentVersion = `reindexed-v${_version.versionService.getMajorVersion()}`;
  return indexName.startsWith('.') ? `.${currentVersion}-${sourceName.substr(1)}` : `${currentVersion}-${sourceName}`;
};
exports.generateNewIndexName = generateNewIndexName;
const getDeprecatedSettingWarning = flatSettings => {
  const {
    settings
  } = flatSettings;
  const deprecatedSettingsInUse = Object.keys(settings || {}).filter(setting => {
    return deprecatedSettings.indexOf(setting) > -1;
  });

  // Translog settings are only marked as deprecated if soft deletes is enabled
  // @ts-expect-error @elastic/elasticsearch doesn't declare such a setting
  if (settings['index.soft_deletes.enabled'] === 'true') {
    // @ts-expect-error @elastic/elasticsearch doesn't declare such a setting
    if (settings['index.translog.retention.size']) {
      deprecatedSettingsInUse.push('index.translog.retention.size');
    }

    // @ts-expect-error @elastic/elasticsearch doesn't declare such a setting
    if (settings['index.translog.retention.age']) {
      deprecatedSettingsInUse.push('index.translog.retention.age');
    }
  }
  if (deprecatedSettingsInUse.length) {
    return {
      flow: 'all',
      warningType: 'indexSetting',
      meta: {
        deprecatedSettings: deprecatedSettingsInUse
      }
    };
  }
};

/**
 * Returns an array of warnings that should be displayed to user before reindexing begins.
 * @param flatSettings
 */
exports.getDeprecatedSettingWarning = getDeprecatedSettingWarning;
const getReindexWarnings = flatSettings => {
  const warnings = [];
  if (_version.versionService.getMajorVersion() === 8) {
    const deprecatedSettingWarning = getDeprecatedSettingWarning(flatSettings);
    if (deprecatedSettingWarning) {
      warnings.push(deprecatedSettingWarning);
    }
  }
  return warnings;
};
exports.getReindexWarnings = getReindexWarnings;