"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDeprecationLoggingStatus = getDeprecationLoggingStatus;
exports.getRecentEsDeprecationLogs = getRecentEsDeprecationLogs;
exports.isDeprecationLogIndexingEnabled = isDeprecationLogIndexingEnabled;
exports.isDeprecationLoggingEnabled = isDeprecationLoggingEnabled;
exports.setDeprecationLogging = setDeprecationLogging;
var _lodash = require("lodash");
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getDeprecationLoggingStatus(dataClient) {
  const response = await dataClient.asCurrentUser.cluster.getSettings({
    include_defaults: true
  });
  return {
    isDeprecationLogIndexingEnabled: isDeprecationLogIndexingEnabled(response),
    isDeprecationLoggingEnabled: isDeprecationLoggingEnabled(response)
  };
}
async function setDeprecationLogging(dataClient, isEnabled) {
  const response = await dataClient.asCurrentUser.cluster.putSettings({
    body: {
      persistent: {
        'logger.deprecation': isEnabled ? 'WARN' : 'ERROR',
        'cluster.deprecation_indexing.enabled': isEnabled
      },
      /*
       * If we only set the persistent setting, we can end up in a situation in which a user has
       * set transient on/off. And when toggling and reloading the page the transient setting will
       * have priority over it thus "overriding" whatever the user selected.
       */
      transient: {
        'logger.deprecation': isEnabled ? 'WARN' : 'ERROR',
        'cluster.deprecation_indexing.enabled': isEnabled
      }
    }
  });
  return {
    isDeprecationLogIndexingEnabled: isEnabled,
    isDeprecationLoggingEnabled: isDeprecationLoggingEnabled(response)
  };
}
async function getRecentEsDeprecationLogs(dataClient, timeframeMsec = _constants.RECENT_DURATION_MS) {
  const indexExists = await dataClient.asCurrentUser.indices.exists({
    index: _constants.DEPRECATION_LOGS_INDEX
  });
  if (!indexExists) {
    return {
      logs: [],
      count: 0
    };
  }
  const now = (0, _momentTimezone.default)();
  const from = (0, _momentTimezone.default)().subtract(timeframeMsec, 'milliseconds');
  try {
    var _searchResponse$hits$;
    const searchResponse = await dataClient.asCurrentUser.search({
      index: _constants.DEPRECATION_LOGS_INDEX,
      body: {
        query: {
          bool: {
            must: {
              range: {
                '@timestamp': {
                  gte: from.toISOString(),
                  lte: now.toISOString()
                }
              }
            },
            must_not: {
              terms: {
                [_constants.DEPRECATION_LOGS_ORIGIN_FIELD]: [..._constants.APPS_WITH_DEPRECATION_LOGS]
              }
            }
          }
        },
        sort: [{
          '@timestamp': {
            order: 'desc'
          }
        }]
      }
    });
    const logs = searchResponse.hits.hits.map(hit => hit._source);
    return {
      logs,
      count: (_searchResponse$hits$ = searchResponse.hits.total) === null || _searchResponse$hits$ === void 0 ? void 0 : _searchResponse$hits$.value
    };
  } catch (error) {
    // If search fails, return empty results to avoid blocking the overall status
    return {
      logs: [],
      count: 0
    };
  }
}
function isDeprecationLogIndexingEnabled(settings) {
  const clusterDeprecationLoggingEnabled = ['defaults', 'persistent', 'transient'].reduce((currentLogLevel, settingsTier) => (0, _lodash.get)(settings, [settingsTier, 'cluster', 'deprecation_indexing', 'enabled'], currentLogLevel), 'false');
  return clusterDeprecationLoggingEnabled === 'true';
}
function isDeprecationLoggingEnabled(settings) {
  const deprecationLogLevel = ['defaults', 'persistent', 'transient'].reduce((currentLogLevel, settingsTier) => (0, _lodash.get)(settings, [settingsTier, 'logger', 'deprecation'], currentLogLevel), 'WARN');
  return ['ALL', 'TRACE', 'DEBUG', 'INFO', 'WARN'].includes(deprecationLogLevel);
}