"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ENT_SEARCH_INDEX_PREFIX = exports.ENT_SEARCH_DATASTREAM_PREFIXES = exports.ENT_SEARCH_DATASTREAM_PATTERN = void 0;
exports.getPreEightEnterpriseSearchIndices = getPreEightEnterpriseSearchIndices;
exports.setPreEightEnterpriseSearchIndicesReadOnly = setPreEightEnterpriseSearchIndicesReadOnly;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ENT_SEARCH_INDEX_PREFIX = exports.ENT_SEARCH_INDEX_PREFIX = '.ent-search-';
const ENT_SEARCH_DATASTREAM_PREFIXES = exports.ENT_SEARCH_DATASTREAM_PREFIXES = ['logs-enterprise_search.', 'logs-app_search.', 'logs-workplace_search.'];
const ENT_SEARCH_DATASTREAM_PATTERN = exports.ENT_SEARCH_DATASTREAM_PATTERN = ['logs-enterprise_search.*', 'logs-app_search.*', 'logs-workplace_search.*'];
function is7xIncompatibleIndex(indexData) {
  var _indexData$settings$i, _indexData$settings, _indexData$settings$i2, _indexData$settings2, _indexData$settings2$, _indexData$settings2$2, _indexData$settings2$3;
  const isReadOnly = (_indexData$settings$i = (_indexData$settings = indexData.settings) === null || _indexData$settings === void 0 ? void 0 : (_indexData$settings$i2 = _indexData$settings.index) === null || _indexData$settings$i2 === void 0 ? void 0 : _indexData$settings$i2.verified_read_only) !== null && _indexData$settings$i !== void 0 ? _indexData$settings$i : 'false';
  return Boolean(((_indexData$settings2 = indexData.settings) === null || _indexData$settings2 === void 0 ? void 0 : (_indexData$settings2$ = _indexData$settings2.index) === null || _indexData$settings2$ === void 0 ? void 0 : (_indexData$settings2$2 = _indexData$settings2$.version) === null || _indexData$settings2$2 === void 0 ? void 0 : (_indexData$settings2$3 = _indexData$settings2$2.created) === null || _indexData$settings2$3 === void 0 ? void 0 : _indexData$settings2$3.startsWith('7')) && isReadOnly !== 'true');
}
async function getPreEightEnterpriseSearchIndices(esClient) {
  const entSearchIndices = await esClient.indices.get({
    index: `${ENT_SEARCH_INDEX_PREFIX}*`,
    ignore_unavailable: true,
    expand_wildcards: ['all', 'hidden']
  });
  const returnIndices = [];
  for (const [index, indexData] of Object.entries(entSearchIndices)) {
    if (is7xIncompatibleIndex(indexData)) {
      const dataStreamName = indexData.data_stream;
      returnIndices.push({
        name: index,
        hasDatastream: dataStreamName ? true : false,
        datastreams: [dataStreamName !== null && dataStreamName !== void 0 ? dataStreamName : '']
      });
    }
  }
  const {
    data_streams: entSearchDatastreams
  } = await esClient.indices.getDataStream({
    name: ENT_SEARCH_DATASTREAM_PATTERN.join(','),
    expand_wildcards: ['all', 'hidden']
  });
  const dsIndices = new Set();
  entSearchDatastreams.forEach(({
    indices: dsi
  }) => {
    dsi.forEach(({
      index_name: indexName
    }) => {
      dsIndices.add(indexName);
    });
  });
  if (!dsIndices.size) return returnIndices;
  for (const returnIndex of returnIndices) {
    if (dsIndices.has(returnIndex.name)) {
      dsIndices.delete(returnIndex.name);
    }
  }
  if (!dsIndices.size) return returnIndices;
  const entSearchDsIndices = await esClient.indices.get({
    index: Array.from(dsIndices.values()),
    ignore_unavailable: true
  });
  for (const [index, indexData] of Object.entries(entSearchDsIndices)) {
    if (is7xIncompatibleIndex(indexData)) {
      const dataStreamName = indexData.data_stream;
      returnIndices.push({
        name: index,
        hasDatastream: dataStreamName ? true : false,
        datastreams: [dataStreamName !== null && dataStreamName !== void 0 ? dataStreamName : '']
      });
    }
  }
  return returnIndices;
}
async function setPreEightEnterpriseSearchIndicesReadOnly(esClient) {
  // get the indices again to ensure nothing's changed since the last check
  let indices = await getPreEightEnterpriseSearchIndices(esClient);

  // rollover any datastreams first
  const rolledOverDatastreams = {};
  for (const index of indices) {
    if (index.hasDatastream) {
      for (const datastream of index.datastreams) {
        if (datastream.length === 0 || rolledOverDatastreams[datastream]) {
          continue;
        }
        const indexResponse = await esClient.indices.rollover({
          alias: datastream
        });
        if (!indexResponse) {
          return `Could not roll over datastream: ${index.name}`;
        }
        rolledOverDatastreams[datastream] = true;
      }
    }
  }
  if (Object.keys(rolledOverDatastreams).length > 0) {
    // we rolled over at least one datastream,
    // get the indices again
    indices = await getPreEightEnterpriseSearchIndices(esClient);
  }
  for (const index of indices) {
    const indexName = index.name;
    const indexResponse = await esClient.indices.addBlock({
      index: indexName,
      block: 'write'
    });
    if (!indexResponse || indexResponse.acknowledged !== true) {
      return `Could not set index read-only: ${indexName}`;
    }
  }
  return '';
}