"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskStore = void 0;
exports.correctVersionConflictsForContinuation = correctVersionConflictsForContinuation;
exports.partialTaskInstanceToAttributes = partialTaskInstanceToAttributes;
exports.savedObjectToConcreteTaskInstance = savedObjectToConcreteTaskInstance;
exports.taskInstanceToAttributes = taskInstanceToAttributes;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _murmurhash = _interopRequireDefault(require("murmurhash"));
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _esQuery = require("@kbn/es-query");
var _result_type = require("./lib/result_type");
var _task = require("./task");
var _task_validator = require("./task_validator");
var _mark_available_tasks_as_claimed = require("./queries/mark_available_tasks_as_claimed");
var _task_partitioner = require("./lib/task_partitioner");
var _msearch_error = require("./lib/msearch_error");
var _bulk_update_error = require("./lib/bulk_update_error");
var _saved_objects = require("./saved_objects");
var _api_key_utils = require("./lib/api_key_utils");
var _get_first_run_at = require("./lib/get_first_run_at");
var _intervals = require("./lib/intervals");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * This module contains helpers for managing the task manager storage layer.
 */

/**
 * Wraps an elasticsearch connection and provides a task manager-specific
 * interface into the index.
 */
class TaskStore {
  /**
   * Constructs a new TaskStore.
   * @param {StoreOpts} opts
   * @prop {esClient} esClient - An elasticsearch client
   * @prop {string} index - The name of the task manager index
   * @prop {TaskDefinition} definition - The definition of the task being run
   * @prop {serializer} - The saved object serializer
   * @prop {savedObjectsRepository} - An instance to the saved objects repository
   */
  constructor(opts) {
    (0, _defineProperty2.default)(this, "index", void 0);
    (0, _defineProperty2.default)(this, "taskManagerId", void 0);
    (0, _defineProperty2.default)(this, "errors$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "taskValidator", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "esoClient", void 0);
    (0, _defineProperty2.default)(this, "esClientWithoutRetries", void 0);
    (0, _defineProperty2.default)(this, "definitions", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsRepository", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsService", void 0);
    (0, _defineProperty2.default)(this, "serializer", void 0);
    (0, _defineProperty2.default)(this, "adHocTaskCounter", void 0);
    (0, _defineProperty2.default)(this, "requestTimeouts", void 0);
    (0, _defineProperty2.default)(this, "security", void 0);
    (0, _defineProperty2.default)(this, "canEncryptSavedObjects", void 0);
    (0, _defineProperty2.default)(this, "getIsSecurityEnabled", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.esClient = opts.esClient;
    this.esoClient = opts.esoClient;
    this.index = opts.index;
    this.taskManagerId = opts.taskManagerId;
    this.definitions = opts.definitions;
    this.serializer = opts.serializer;
    this.savedObjectsRepository = opts.savedObjectsRepository;
    this.savedObjectsService = opts.savedObjectsService;
    this.adHocTaskCounter = opts.adHocTaskCounter;
    this.taskValidator = new _task_validator.TaskValidator({
      logger: opts.logger,
      definitions: opts.definitions,
      allowReadingInvalidState: opts.allowReadingInvalidState
    });
    this.esClientWithoutRetries = opts.esClient.child({
      // Timeouts are retried and make requests timeout after (requestTimeout * (1 + maxRetries))
      // The poller doesn't need retry logic because it will try again at the next polling cycle
      maxRetries: 0
    });
    this.requestTimeouts = opts.requestTimeouts;
    this.security = opts.security;
    this.canEncryptSavedObjects = opts.canEncryptSavedObjects;
    this.getIsSecurityEnabled = opts.getIsSecurityEnabled;
    this.logger = opts.logger;
  }
  registerEncryptedSavedObjectsClient(client) {
    this.esoClient = client;
  }
  canEncryptSo() {
    return !!(this.esoClient && this.canEncryptSavedObjects);
  }
  validateCanEncryptSavedObjects(request) {
    if (!request) {
      return;
    }
    if (!this.canEncryptSo()) {
      throw Error('Unable to schedule task(s) with API keys because the Encrypted Saved Objects plugin has not been registered or is missing encryption key.');
    }
  }
  getSoClientForCreate(options) {
    if (options.request && this.getIsSecurityEnabled()) {
      return this.savedObjectsService.getScopedClient(options.request, {
        includedHiddenTypes: [_saved_objects.TASK_SO_NAME],
        excludedExtensions: [_server.SECURITY_EXTENSION_ID, _server.SPACES_EXTENSION_ID]
      });
    }
    return this.savedObjectsRepository;
  }
  async getApiKeyFromRequest(taskInstances, request) {
    if (!this.getIsSecurityEnabled()) {
      return null;
    }
    if (!request) {
      return null;
    }
    let userScopeAndApiKey;
    try {
      userScopeAndApiKey = await (0, _api_key_utils.getApiKeyAndUserScope)(taskInstances, request, this.security);
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    return userScopeAndApiKey;
  }
  async bulkGetDecryptedTaskApiKeys(ids) {
    const result = new Map();
    if (!this.canEncryptSo() || !ids.length) {
      return result;
    }
    const kueryNode = _esQuery.nodeBuilder.or(ids.map(id => {
      return _esQuery.nodeBuilder.is(`${_saved_objects.TASK_SO_NAME}.id`, `${_saved_objects.TASK_SO_NAME}:${id}`);
    }));
    const finder = await this.esoClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: _saved_objects.TASK_SO_NAME,
      filter: kueryNode
    });
    for await (const response of finder.find()) {
      response.saved_objects.forEach(savedObject => {
        result.set(savedObject.id, savedObject.attributes.apiKey);
      });
    }
    await finder.close();
    return result;
  }
  async bulkGetAndMergeTasksWithDecryptedApiKey(tasks) {
    const ids = [];
    tasks.forEach(task => {
      if (task.apiKey) {
        ids.push(task.id);
      }
    });
    if (!ids.length) {
      return tasks;
    }
    const decryptedTaskApiKeysMap = await this.bulkGetDecryptedTaskApiKeys(ids);
    const tasksWithDecryptedApiKeys = tasks.map(task => ({
      ...task,
      ...(decryptedTaskApiKeysMap.get(task.id) ? {
        apiKey: decryptedTaskApiKeysMap.get(task.id)
      } : {})
    }));
    return tasksWithDecryptedApiKeys;
  }

  /**
   * Convert ConcreteTaskInstance Ids to match their SavedObject format as serialized
   * in Elasticsearch
   * @param tasks - The task being scheduled.
   */
  convertToSavedObjectIds(taskIds) {
    return taskIds.map(id => this.serializer.generateRawId(undefined, 'task', id));
  }

  /**
   * Schedules a task.
   *
   * @param task - The task being scheduled.
   */
  async schedule(taskInstance, options) {
    try {
      this.validateCanEncryptSavedObjects(options === null || options === void 0 ? void 0 : options.request);
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    this.definitions.ensureHas(taskInstance.taskType);
    const apiKeyAndUserScopeMap = (await this.getApiKeyFromRequest([taskInstance], options === null || options === void 0 ? void 0 : options.request)) || new Map();
    const {
      apiKey,
      userScope
    } = apiKeyAndUserScopeMap.get(taskInstance.id) || {};
    const soClient = this.getSoClientForCreate(options || {});
    let savedObject;
    try {
      const id = taskInstance.id || (0, _uuid.v4)();
      const validatedTaskInstance = this.taskValidator.getValidatedTaskInstanceForUpdating(taskInstance);
      savedObject = await soClient.create('task', {
        ...taskInstanceToAttributes(validatedTaskInstance, id),
        ...(userScope ? {
          userScope
        } : {}),
        ...(apiKey ? {
          apiKey
        } : {}),
        runAt: (0, _get_first_run_at.getFirstRunAt)({
          taskInstance: validatedTaskInstance,
          logger: this.logger
        })
      }, {
        id,
        refresh: false
      });
      if ((0, _lodash.get)(taskInstance, 'schedule.interval', null) == null && (0, _lodash.get)(taskInstance, 'schedule.rrule', null) == null) {
        this.adHocTaskCounter.increment();
      }
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    const result = savedObjectToConcreteTaskInstance(savedObject);
    if (options !== null && options !== void 0 && options.request && !this.getIsSecurityEnabled()) {
      this.logger.info(`Trying to schedule task ${result.id} with user scope but security is disabled. Task will run without user scope.`);
    }
    return this.taskValidator.getValidatedTaskInstanceFromReading(result);
  }

  /**
   * Bulk schedules a task.
   *
   * @param tasks - The tasks being scheduled.
   */
  async bulkSchedule(taskInstances, options) {
    try {
      this.validateCanEncryptSavedObjects(options === null || options === void 0 ? void 0 : options.request);
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    const apiKeyAndUserScopeMap = (await this.getApiKeyFromRequest(taskInstances, options === null || options === void 0 ? void 0 : options.request)) || new Map();
    const soClient = this.getSoClientForCreate(options || {});
    const objects = taskInstances.reduce((acc, taskInstance) => {
      const {
        apiKey,
        userScope
      } = apiKeyAndUserScopeMap.get(taskInstance.id) || {};
      const id = taskInstance.id || (0, _uuid.v4)();
      this.definitions.ensureHas(taskInstance.taskType);
      try {
        const validatedTaskInstance = this.taskValidator.getValidatedTaskInstanceForUpdating(taskInstance);
        return [...acc, {
          type: 'task',
          attributes: {
            ...taskInstanceToAttributes(validatedTaskInstance, id),
            ...(apiKey ? {
              apiKey
            } : {}),
            ...(userScope ? {
              userScope
            } : {}),
            runAt: (0, _get_first_run_at.getFirstRunAt)({
              taskInstance: validatedTaskInstance,
              logger: this.logger
            })
          },
          id
        }];
      } catch (e) {
        this.logger.error(`[TaskStore] An error occured. Task ${taskInstance.id} will not be updated. Error: ${e.message}`);
        return acc;
      }
    }, []);
    let savedObjects;
    try {
      savedObjects = await soClient.bulkCreate(objects, {
        refresh: false,
        overwrite: true
      });
      this.adHocTaskCounter.increment(taskInstances.filter(task => {
        return (0, _lodash.get)(task, 'schedule.interval', null) == null;
      }).length);
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    if (options !== null && options !== void 0 && options.request && !this.getIsSecurityEnabled()) {
      this.logger.info(`Trying to bulk schedule tasks ${JSON.stringify(savedObjects.saved_objects.map(so => so.id))} with user scope but security is disabled. Tasks will run without user scope.`);
    }
    return savedObjects.saved_objects.map(so => {
      const taskInstance = savedObjectToConcreteTaskInstance(so);
      return this.taskValidator.getValidatedTaskInstanceFromReading(taskInstance);
    });
  }

  /**
   * Fetches a list of scheduled tasks with default sorting.
   *
   * @param opts - The query options used to filter tasks
   * @param limitResponse - Whether to exclude the task state and params from the source for a smaller respose payload
   */
  async fetch({
    sort = [{
      'task.runAt': 'asc'
    }],
    ...opts
  } = {}, limitResponse = false) {
    return this.search({
      ...opts,
      sort
    }, limitResponse);
  }

  /**
   * Updates the specified doc in the index, returning the doc
   * with its version up to date.
   *
   * @param {TaskDoc} doc
   * @returns {Promise<TaskDoc>}
   */
  async update(doc, options) {
    let updatedSavedObject;
    let attributes;
    try {
      const taskInstance = this.taskValidator.getValidatedTaskInstanceForUpdating(doc, {
        validate: options.validate
      });
      attributes = taskInstanceToAttributes(taskInstance, doc.id);
      updatedSavedObject = await this.savedObjectsRepository.update('task', doc.id, attributes, {
        refresh: false,
        version: doc.version
      });
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    const result = savedObjectToConcreteTaskInstance(
    // The SavedObjects update api forces a Partial on the `attributes` on the response,
    // but actually returns the whole object that is passed to it, so as we know we're
    // passing in the whole object, this is safe to do.
    // This is far from ideal, but unless we change the SavedObjectsClient this is the best we can do
    {
      ...updatedSavedObject,
      attributes: (0, _lodash.defaults)(updatedSavedObject.attributes, attributes)
    });
    return this.taskValidator.getValidatedTaskInstanceFromReading(result, {
      validate: options.validate
    });
  }

  /**
   * Updates the specified docs in the index, returning the docs
   * with their versions up to date.
   *
   * @param {Array<TaskDoc>} docs
   * @returns {Promise<Array<TaskDoc>>}
   */
  async bulkUpdate(docs, {
    validate
  }) {
    const newDocs = docs.reduce((acc, doc) => {
      try {
        const taskInstance = this.taskValidator.getValidatedTaskInstanceForUpdating(doc, {
          validate
        });
        acc.set(doc.id, {
          type: 'task',
          id: doc.id,
          version: doc.version,
          attributes: taskInstanceToAttributes(taskInstance, doc.id)
        });
      } catch (e) {
        this.logger.error(`[TaskStore] An error occured. Task ${doc.id} will not be updated. Error: ${e.message}`);
      }
      return acc;
    }, new Map());
    let updatedSavedObjects;
    try {
      ({
        saved_objects: updatedSavedObjects
      } = await this.savedObjectsRepository.bulkUpdate(Array.from(newDocs.values()), {
        refresh: false
      }));
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    return updatedSavedObjects.map(updatedSavedObject => {
      var _newDocs$get;
      if (updatedSavedObject.error !== undefined) {
        return (0, _result_type.asErr)({
          type: 'task',
          id: updatedSavedObject.id,
          error: updatedSavedObject.error
        });
      }
      const taskInstance = savedObjectToConcreteTaskInstance({
        ...updatedSavedObject,
        attributes: (0, _lodash.defaults)(updatedSavedObject.attributes, (_newDocs$get = newDocs.get(updatedSavedObject.id)) === null || _newDocs$get === void 0 ? void 0 : _newDocs$get.attributes)
      });
      const result = this.taskValidator.getValidatedTaskInstanceFromReading(taskInstance, {
        validate
      });
      return (0, _result_type.asOk)(result);
    });
  }
  async bulkPartialUpdate(docs) {
    if (docs.length === 0) {
      return [];
    }
    const bulkBody = [];
    for (const doc of docs) {
      var _doc$schedule;
      if ((_doc$schedule = doc.schedule) !== null && _doc$schedule !== void 0 && _doc$schedule.interval && !(0, _intervals.isInterval)(doc.schedule.interval)) {
        this.logger.error(`[TaskStore] Invalid interval "${doc.schedule.interval}". Task ${doc.id} will not be updated.`);
        continue;
      }
      bulkBody.push({
        update: {
          _id: `task:${doc.id}`,
          ...(doc.version ? (0, _coreSavedObjectsBaseServerInternal.decodeRequestVersion)(doc.version) : {})
        }
      });
      bulkBody.push({
        doc: {
          task: partialTaskInstanceToAttributes(doc)
        }
      });
    }
    let result;
    try {
      result = await this.esClient.bulk({
        index: this.index,
        refresh: false,
        body: bulkBody
      });
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    return result.items.map(item => {
      var _item$update;
      const malformedResponseType = 'malformed response';
      if (!item.update || !item.update._id) {
        const err = new _bulk_update_error.BulkUpdateError({
          message: malformedResponseType,
          type: malformedResponseType,
          statusCode: 500
        });
        this.errors$.next(err);
        return (0, _result_type.asErr)({
          type: 'task',
          id: 'unknown',
          error: {
            type: malformedResponseType
          }
        });
      }
      const docId = item.update._id.startsWith('task:') ? item.update._id.slice(5) : item.update._id;
      if ((_item$update = item.update) !== null && _item$update !== void 0 && _item$update.error) {
        const err = new _bulk_update_error.BulkUpdateError({
          message: item.update.error.reason,
          type: item.update.error.type,
          statusCode: item.update.status
        });
        this.errors$.next(err);
        return (0, _result_type.asErr)({
          type: 'task',
          id: docId,
          status: item.update.status,
          error: item.update.error
        });
      }
      const doc = docs.find(d => d.id === docId);
      return (0, _result_type.asOk)({
        ...doc,
        id: docId,
        version: (0, _coreSavedObjectsBaseServerInternal.encodeVersion)(item.update._seq_no, item.update._primary_term)
      });
    });
  }

  /**
   * Removes the specified task from the index.
   *
   * @param {string} id
   * @returns {Promise<void>}
   */
  async remove(id) {
    const taskInstance = await this.get(id);
    const {
      apiKey,
      userScope
    } = taskInstance;
    if (apiKey && userScope) {
      if (!userScope.apiKeyCreatedByUser) {
        await this.security.authc.apiKeys.invalidateAsInternalUser({
          ids: [userScope.apiKeyId]
        });
      }
    }
    try {
      await this.savedObjectsRepository.delete('task', id, {
        refresh: false
      });
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
  }

  /**
   * Bulk removes the specified tasks from the index.
   *
   * @param {SavedObjectsBulkDeleteObject[]} savedObjectsToDelete
   * @returns {Promise<SavedObjectsBulkDeleteResponse>}
   */
  async bulkRemove(taskIds) {
    const taskInstances = await this.bulkGet(taskIds);
    const apiKeyIdsToRemove = [];
    taskInstances.forEach(taskInstance => {
      const unwrappedTaskInstance = (0, _result_type.unwrap)(taskInstance);
      const {
        apiKey,
        userScope
      } = unwrappedTaskInstance;
      if (apiKey && userScope) {
        if (!userScope.apiKeyCreatedByUser) {
          apiKeyIdsToRemove.push(userScope.apiKeyId);
        }
      }
    });
    if (apiKeyIdsToRemove.length) {
      await this.security.authc.apiKeys.invalidateAsInternalUser({
        ids: [...new Set(apiKeyIdsToRemove)]
      });
    }
    try {
      const savedObjectsToDelete = taskIds.map(taskId => ({
        id: taskId,
        type: 'task'
      }));
      return await this.savedObjectsRepository.bulkDelete(savedObjectsToDelete, {
        refresh: false
      });
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
  }

  /**
   * Gets a task by id
   *
   * @param {string} id
   * @returns {Promise<void>}
   */
  async get(id) {
    let result;
    try {
      result = await this.savedObjectsRepository.get('task', id);
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    const taskInstance = savedObjectToConcreteTaskInstance(result);
    const tasksWithDecryptedApiKeys = await this.bulkGetAndMergeTasksWithDecryptedApiKey([taskInstance]);
    return tasksWithDecryptedApiKeys[0];
  }

  /**
   * Gets tasks by ids
   *
   * @param {Array<string>} ids
   * @returns {Promise<ConcreteTaskInstance[]>}
   */
  async bulkGet(ids) {
    let result;
    try {
      result = await this.savedObjectsRepository.bulkGet(ids.map(id => ({
        type: 'task',
        id
      })));
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    const tasks = [];
    result.saved_objects.forEach(task => {
      if (!task.error) {
        tasks.push(savedObjectToConcreteTaskInstance(task));
      }
    });
    const tasksWithDecryptedApiKeys = await this.bulkGetAndMergeTasksWithDecryptedApiKey(tasks);
    const taskMap = new Map();
    tasksWithDecryptedApiKeys.forEach(task => taskMap.set(task.id, task));
    return result.saved_objects.map(task => {
      if (task.error) {
        return (0, _result_type.asErr)({
          id: task.id,
          type: task.type,
          error: task.error
        });
      }
      return (0, _result_type.asOk)(taskMap.get(task.id));
    });
  }

  /**
   * Gets task version info by ids
   *
   * @param {Array<string>} esIds
   * @returns {Promise<ConcreteTaskInstance[]>}
   */
  async bulkGetVersions(ids) {
    let taskVersions;
    try {
      taskVersions = await this.esClientWithoutRetries.mget({
        index: this.index,
        _source: false,
        body: {
          ids
        }
      });
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
    const result = taskVersions.docs.map(taskVersion => {
      var _taskVersion$error, _taskVersion$error2;
      if (isMGetSuccess(taskVersion)) {
        if (!taskVersion.found) {
          return {
            esId: taskVersion._id,
            error: `task "${taskVersion._id}" not found`
          };
        } else {
          return {
            esId: taskVersion._id,
            seqNo: taskVersion._seq_no,
            primaryTerm: taskVersion._primary_term
          };
        }
      }
      const type = ((_taskVersion$error = taskVersion.error) === null || _taskVersion$error === void 0 ? void 0 : _taskVersion$error.type) || 'unknown type of error';
      const reason = ((_taskVersion$error2 = taskVersion.error) === null || _taskVersion$error2 === void 0 ? void 0 : _taskVersion$error2.reason) || 'unknown reason';
      const error = `error getting version for ${taskVersion._id}: ${type}: ${reason}`;
      return {
        esId: taskVersion._id,
        error
      };
    });
    return result;
  }

  /**
   * Gets task lifecycle step by id
   *
   * @param {string} id
   * @returns {Promise<void>}
   */
  async getLifecycle(id) {
    try {
      const task = await this.get(id);
      return task.status;
    } catch (err) {
      if (err.output && err.output.statusCode === 404) {
        return _task.TaskLifecycleResult.NotFound;
      }
      throw err;
    }
  }

  // like search(), only runs multiple searches in parallel returning the combined results
  async msearch(opts = []) {
    const queries = opts.map(({
      sort = [{
        'task.runAt': 'asc'
      }],
      ...opt
    }) => ensureQueryOnlyReturnsTaskObjects({
      sort,
      ...opt
    }));
    const body = queries.flatMap(query => [{}, query]);
    const result = await this.esClientWithoutRetries.msearch({
      index: this.index,
      ignore_unavailable: true,
      body
    });
    const {
      responses
    } = result;
    const versionMap = this.createVersionMap([]);
    let allTasks = new Array();
    for (const response of responses) {
      if (response.status !== 200) {
        const err = new _msearch_error.MsearchError(response.status);
        this.errors$.next(err);
        throw err;
      }
      const {
        hits
      } = response;
      const {
        hits: tasks
      } = hits;
      this.addTasksToVersionMap(versionMap, tasks);
      allTasks = allTasks.concat(this.filterTasks(tasks));
    }
    const allSortedTasks = (0, _mark_available_tasks_as_claimed.claimSort)(this.definitions, allTasks);
    const tasksWithDecryptedApiKeys = await this.bulkGetAndMergeTasksWithDecryptedApiKey(allSortedTasks);
    return {
      docs: tasksWithDecryptedApiKeys,
      versionMap
    };
  }
  async search(opts = {}, limitResponse = false) {
    const {
      query
    } = ensureQueryOnlyReturnsTaskObjects(opts);
    try {
      const result = await this.esClientWithoutRetries.search({
        index: this.index,
        ignore_unavailable: true,
        body: {
          ...opts,
          query
        },
        ...(limitResponse ? {
          _source_excludes: ['task.state', 'task.params']
        } : {})
      });
      const {
        hits: {
          hits: tasks
        }
      } = result;
      const versionMap = this.createVersionMap(tasks);
      const concreteTasks = this.filterTasks(tasks);
      const tasksWithDecryptedApiKeys = await this.bulkGetAndMergeTasksWithDecryptedApiKey(concreteTasks);
      return {
        docs: tasksWithDecryptedApiKeys,
        versionMap
      };
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
  }
  filterTasks(tasks) {
    return tasks
    // @ts-expect-error @elastic/elasticsearch _source is optional
    .filter(doc => this.serializer.isRawSavedObject(doc))
    // @ts-expect-error @elastic/elasticsearch _source is optional
    .map(doc => this.serializer.rawToSavedObject(doc)).map(doc => (0, _lodash.omit)(doc, 'namespace')).map(doc => savedObjectToConcreteTaskInstance(doc)).filter(doc => !!doc);
  }
  addTasksToVersionMap(versionMap, tasks) {
    for (const task of tasks) {
      if (task._id == null || task._seq_no == null || task._primary_term == null) continue;
      const esId = task._id.startsWith('task:') ? task._id.slice(5) : task._id;
      versionMap.set(esId, {
        esId: task._id,
        seqNo: task._seq_no,
        primaryTerm: task._primary_term
      });
    }
  }
  createVersionMap(tasks) {
    const versionMap = new Map();
    this.addTasksToVersionMap(versionMap, tasks);
    return versionMap;
  }
  async aggregate({
    aggs,
    query,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    runtime_mappings,
    size = 0
  }) {
    const body = await this.esClient.search({
      index: this.index,
      ignore_unavailable: true,
      track_total_hits: true,
      body: ensureAggregationOnlyReturnsEnabledTaskObjects({
        query,
        aggs,
        runtime_mappings,
        size
      })
    });
    return body;
  }
  async updateByQuery(opts = {},
  // eslint-disable-next-line @typescript-eslint/naming-convention
  {
    max_docs: max_docs
  } = {}) {
    const {
      query
    } = ensureQueryOnlyReturnsTaskObjects(opts);
    try {
      const
      // eslint-disable-next-line @typescript-eslint/naming-convention
      {
        total,
        updated,
        version_conflicts
      } = await this.esClientWithoutRetries.updateByQuery({
        index: this.index,
        ignore_unavailable: true,
        refresh: true,
        conflicts: 'proceed',
        body: {
          ...opts,
          max_docs,
          query
        }
      }, {
        requestTimeout: this.requestTimeouts.update_by_query
      });
      const conflictsCorrectedForContinuation = correctVersionConflictsForContinuation(updated, version_conflicts, max_docs);
      return {
        total: total || 0,
        updated: updated || 0,
        version_conflicts: conflictsCorrectedForContinuation
      };
    } catch (e) {
      this.errors$.next(e);
      throw e;
    }
  }
  async getDocVersions(esIds) {
    const versions = await this.bulkGetVersions(esIds);
    const result = new Map();
    for (const version of versions) {
      result.set(version.esId, version);
    }
    return result;
  }
}

/**
 * When we run updateByQuery with conflicts='proceed', it's possible for the `version_conflicts`
 * to count against the specified `max_docs`, as per https://github.com/elastic/elasticsearch/issues/63671
 * In order to correct for that happening, we only count `version_conflicts` if we haven't updated as
 * many docs as we could have.
 * This is still no more than an estimation, as there might have been less docuemnt to update that the
 * `max_docs`, but we bias in favour of over zealous `version_conflicts` as that's the best indicator we
 * have for an unhealthy cluster distribution of Task Manager polling intervals
 */
exports.TaskStore = TaskStore;
function correctVersionConflictsForContinuation(updated, versionConflicts, maxDocs) {
  // @ts-expect-error estypes.ReindexResponse['updated'] and estypes.ReindexResponse['version_conflicts'] can be undefined
  return maxDocs && versionConflicts + updated > maxDocs ? maxDocs - updated : versionConflicts;
}
function taskInstanceToAttributes(doc, id) {
  return {
    ...(0, _lodash.omit)(doc, 'id', 'version', 'userScope', 'apiKey'),
    params: JSON.stringify(doc.params || {}),
    state: JSON.stringify(doc.state || {}),
    attempts: doc.attempts || 0,
    scheduledAt: (doc.scheduledAt || new Date()).toISOString(),
    startedAt: doc.startedAt && doc.startedAt.toISOString() || null,
    retryAt: doc.retryAt && doc.retryAt.toISOString() || null,
    runAt: (doc.runAt || new Date()).toISOString(),
    status: doc.status || 'idle',
    partition: doc.partition || _murmurhash.default.v3(id) % _task_partitioner.MAX_PARTITIONS
  };
}
function partialTaskInstanceToAttributes(doc) {
  return {
    ...(0, _lodash.omit)(doc, 'id', 'version', 'userScope', 'apiKey'),
    ...(doc.params ? {
      params: JSON.stringify(doc.params)
    } : {}),
    ...(doc.state ? {
      state: JSON.stringify(doc.state)
    } : {}),
    ...(doc.scheduledAt ? {
      scheduledAt: doc.scheduledAt.toISOString()
    } : {}),
    ...(doc.startedAt ? {
      startedAt: doc.startedAt.toISOString()
    } : {}),
    ...(doc.retryAt ? {
      retryAt: doc.retryAt.toISOString()
    } : {}),
    ...(doc.runAt ? {
      runAt: doc.runAt.toISOString()
    } : {})
  };
}
function savedObjectToConcreteTaskInstance(savedObject) {
  return {
    ...savedObject.attributes,
    id: savedObject.id,
    version: savedObject.version,
    scheduledAt: new Date(savedObject.attributes.scheduledAt),
    runAt: new Date(savedObject.attributes.runAt),
    startedAt: savedObject.attributes.startedAt ? new Date(savedObject.attributes.startedAt) : null,
    retryAt: savedObject.attributes.retryAt ? new Date(savedObject.attributes.retryAt) : null,
    state: parseJSONField(savedObject.attributes.state, 'state', savedObject.id),
    params: parseJSONField(savedObject.attributes.params, 'params', savedObject.id)
  };
}
function parseJSONField(json, fieldName, id) {
  try {
    return json ? JSON.parse(json) : {};
  } catch (error) {
    throw new Error(`Task "${id}"'s ${fieldName} field has invalid JSON: ${json}`);
  }
}
function ensureQueryOnlyReturnsTaskObjects(opts) {
  const originalQuery = opts.query;
  const queryOnlyTasks = {
    term: {
      type: 'task'
    }
  };
  const query = originalQuery ? {
    bool: {
      must: [queryOnlyTasks, originalQuery]
    }
  } : queryOnlyTasks;
  return {
    ...opts,
    query
  };
}
function ensureAggregationOnlyReturnsEnabledTaskObjects(opts) {
  const originalQuery = opts.query;
  const filterToOnlyTasks = {
    bool: {
      filter: {
        bool: {
          must: [{
            term: {
              type: 'task'
            }
          }, {
            term: {
              'task.enabled': true
            }
          }],
          must_not: [{
            term: {
              'task.status': _task.TaskStatus.Unrecognized
            }
          }]
        }
      }
    }
  };
  const query = originalQuery ? {
    bool: {
      must: [filterToOnlyTasks, originalQuery]
    }
  } : filterToOnlyTasks;
  return {
    ...opts,
    query
  };
}
function isMGetSuccess(doc) {
  return doc.found !== undefined;
}