"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TASK_ID = exports.CLEANUP_LOOKBACK = exports.CLEANUP_INTERVAL = void 0;
exports.registerDeleteInactiveNodesTaskDefinition = registerDeleteInactiveNodesTaskDefinition;
exports.scheduleDeleteInactiveNodesTaskDefinition = scheduleDeleteInactiveNodesTaskDefinition;
exports.taskRunner = taskRunner;
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_ID = exports.TASK_ID = 'delete_inactive_background_task_nodes';
const TASK_TYPE = `task_manager:${TASK_ID}`;
const CLEANUP_INTERVAL = exports.CLEANUP_INTERVAL = '1m';
const CLEANUP_LOOKBACK = exports.CLEANUP_LOOKBACK = '5m';
async function scheduleDeleteInactiveNodesTaskDefinition(logger, taskScheduling) {
  try {
    await taskScheduling.ensureScheduled({
      id: TASK_ID,
      taskType: TASK_TYPE,
      schedule: {
        interval: CLEANUP_INTERVAL
      },
      state: {},
      params: {}
    });
  } catch (e) {
    logger.error(`Error scheduling ${TASK_ID} task, received ${e.message}`);
  }
}
function registerDeleteInactiveNodesTaskDefinition(logger, coreStartServices, taskTypeDictionary) {
  taskTypeDictionary.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Remove inactive background task nodes',
      createTaskRunner: taskRunner(logger, coreStartServices)
    }
  });
}
function taskRunner(logger, coreStartServices) {
  return () => {
    return {
      async run() {
        try {
          const [{
            savedObjects
          }] = await coreStartServices();
          const savedObjectsRepository = savedObjects.createInternalRepository([_saved_objects.BACKGROUND_TASK_NODE_SO_NAME]);
          const {
            saved_objects: inactiveNodes
          } = await savedObjectsRepository.find({
            type: _saved_objects.BACKGROUND_TASK_NODE_SO_NAME,
            perPage: 100,
            page: 1,
            filter: `${_saved_objects.BACKGROUND_TASK_NODE_SO_NAME}.attributes.last_seen < now-${CLEANUP_LOOKBACK}`
          });
          if (inactiveNodes.length > 0) {
            const nodesToDelete = inactiveNodes.map(node => ({
              type: _saved_objects.BACKGROUND_TASK_NODE_SO_NAME,
              id: node.attributes.id
            }));
            await savedObjectsRepository.bulkDelete(nodesToDelete, {
              force: true,
              refresh: false
            });
            const deletedNodes = nodesToDelete.map(node => node.id);
            logger.debug(`Inactive Kibana nodes: ${deletedNodes}, have been successfully deleted`);
          }
          return {
            state: {},
            schedule: {
              interval: CLEANUP_INTERVAL
            }
          };
        } catch (e) {
          logger.error(`Deleting inactive nodes failed. Error: ${e.message} `);
          return {
            state: {},
            schedule: {
              interval: CLEANUP_INTERVAL
            }
          };
        }
      }
    };
  };
}