"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runSynPrivateLocationMonitorsTaskSoon = exports.SyncPrivateLocationMonitorsTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/alerting-plugin/common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _saved_objects = require("../../common/types/saved_objects");
var _utils = require("../synthetics_service/utils");
var _monitor_config_repository = require("../services/monitor_config_repository");
var _get_private_locations = require("../synthetics_service/get_private_locations");
var _format_configs = require("../synthetics_service/formatters/public_formatters/format_configs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'Synthetics:Sync-Private-Location-Monitors';
const TASK_ID = `${TASK_TYPE}-single-instance`;
const TASK_SCHEDULE = '5m';
class SyncPrivateLocationMonitorsTask {
  constructor(serverSetup, _taskManager, syntheticsMonitorClient) {
    (0, _defineProperty2.default)(this, "start", async () => {
      const {
        logger,
        pluginsStart: {
          taskManager
        }
      } = this.serverSetup;
      logger.debug(`Scheduling private location task`);
      await taskManager.ensureScheduled({
        id: TASK_ID,
        state: {},
        schedule: {
          interval: TASK_SCHEDULE
        },
        taskType: TASK_TYPE,
        params: {}
      });
      logger.debug(`Sync private location monitors task scheduled successfully`);
    });
    (0, _defineProperty2.default)(this, "hasAnyDataChanged", async ({
      taskInstance,
      soClient
    }) => {
      const lastStartedAt = taskInstance.state.lastStartedAt || (0, _moment.default)().subtract(10, 'minute').toISOString();
      const lastTotalParams = taskInstance.state.lastTotalParams || 0;
      const lastTotalMWs = taskInstance.state.lastTotalMWs || 0;
      const {
        totalParams,
        hasParamsChanges
      } = await this.hasAnyParamChanged({
        soClient,
        lastStartedAt,
        lastTotalParams
      });
      const {
        totalMWs,
        hasMWsChanged
      } = await this.hasMWsChanged({
        soClient,
        lastStartedAt,
        lastTotalMWs
      });
      const hasDataChanged = hasMWsChanged || hasParamsChanges;
      return {
        hasDataChanged,
        totalParams,
        totalMWs
      };
    });
    (0, _defineProperty2.default)(this, "debugLog", message => {
      this.serverSetup.logger.debug(`[syncGlobalParams] ${message} `);
    });
    this.serverSetup = serverSetup;
    this.taskManager = _taskManager;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
    _taskManager.registerTaskDefinitions({
      [TASK_TYPE]: {
        title: 'Synthetics Sync Global Params Task',
        description: 'This task is executed so that we can sync private location monitors for example when global params are updated',
        timeout: '3m',
        maxAttempts: 3,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask({
                taskInstance
              });
            }
          };
        }
      }
    });
  }
  async runTask({
    taskInstance
  }) {
    const {
      coreStart: {
        savedObjects
      },
      encryptedSavedObjects,
      logger
    } = this.serverSetup;
    const lastStartedAt = taskInstance.state.lastStartedAt || (0, _moment.default)().subtract(10, 'minute').toISOString();
    const startedAt = taskInstance.startedAt || new Date();
    let lastTotalParams = taskInstance.state.lastTotalParams || 0;
    let lastTotalMWs = taskInstance.state.lastTotalMWs || 0;
    try {
      this.debugLog(`Syncing private location monitors, last total params ${lastTotalParams}`);
      const soClient = savedObjects.createInternalRepository([_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE]);
      const allPrivateLocations = await (0, _get_private_locations.getPrivateLocations)(soClient, _constants.ALL_SPACES_ID);
      const {
        totalMWs,
        totalParams,
        hasDataChanged
      } = await this.hasAnyDataChanged({
        soClient,
        taskInstance
      });
      lastTotalParams = totalParams;
      lastTotalMWs = totalMWs;
      if (hasDataChanged) {
        this.debugLog(`Syncing private location monitors because data has changed`);
        if (allPrivateLocations.length > 0) {
          await this.syncGlobalParams({
            allPrivateLocations,
            soClient,
            encryptedSavedObjects
          });
        }
        this.debugLog(`Sync of private location monitors succeeded`);
      } else {
        this.debugLog(`No data has changed since last run ${lastStartedAt}, skipping sync of private location monitors`);
      }
    } catch (error) {
      logger.error(`Sync of private location monitors failed: ${error.message}`);
      return {
        error,
        state: {
          lastStartedAt: startedAt.toISOString(),
          lastTotalParams,
          lastTotalMWs
        }
      };
    }
    return {
      state: {
        lastStartedAt: startedAt.toISOString(),
        lastTotalParams,
        lastTotalMWs
      }
    };
  }
  async syncGlobalParams({
    allPrivateLocations,
    encryptedSavedObjects,
    soClient
  }) {
    const {
      privateLocationAPI
    } = this.syntheticsMonitorClient;
    const privateConfigs = [];
    const {
      configsBySpaces,
      paramsBySpace,
      spaceIds,
      maintenanceWindows
    } = await this.getAllMonitorConfigs({
      encryptedSavedObjects,
      soClient
    });
    for (const spaceId of spaceIds) {
      var _monitors$length;
      const monitors = configsBySpaces[spaceId];
      this.debugLog(`Processing spaceId: ${spaceId}, monitors count: ${(_monitors$length = monitors === null || monitors === void 0 ? void 0 : monitors.length) !== null && _monitors$length !== void 0 ? _monitors$length : 0}`);
      if (!monitors) {
        continue;
      }
      for (const monitor of monitors) {
        const {
          privateLocations
        } = this.parseLocations(monitor);
        if (privateLocations.length > 0) {
          privateConfigs.push({
            config: monitor,
            globalParams: paramsBySpace[spaceId]
          });
        }
      }
      if (privateConfigs.length > 0) {
        this.debugLog(`Syncing private configs for spaceId: ${spaceId}, privateConfigs count: ${privateConfigs.length}`);
        await privateLocationAPI.editMonitors(privateConfigs, allPrivateLocations, spaceId, maintenanceWindows);
      } else {
        this.debugLog(`No privateConfigs to sync for spaceId: ${spaceId}`);
      }
    }
  }
  async getAllMonitorConfigs({
    soClient,
    encryptedSavedObjects
  }) {
    const {
      syntheticsService
    } = this.syntheticsMonitorClient;
    const paramsBySpacePromise = syntheticsService.getSyntheticsParams({
      spaceId: _constants.ALL_SPACES_ID
    });
    const maintenanceWindowsPromise = syntheticsService.getMaintenanceWindows();
    const monitorConfigRepository = new _monitor_config_repository.MonitorConfigRepository(soClient, encryptedSavedObjects.getClient());
    const monitorsPromise = monitorConfigRepository.findDecryptedMonitors({
      spaceId: _constants.ALL_SPACES_ID
    });
    const [paramsBySpace, monitors, maintenanceWindows] = await Promise.all([paramsBySpacePromise, monitorsPromise, maintenanceWindowsPromise]);
    return {
      ...this.mixParamsWithMonitors(monitors, paramsBySpace),
      paramsBySpace,
      maintenanceWindows
    };
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  mixParamsWithMonitors(monitors, paramsBySpace) {
    const configsBySpaces = {};
    const spaceIds = new Set();
    for (const monitor of monitors) {
      var _monitor$namespaces;
      const spaceId = (_monitor$namespaces = monitor.namespaces) === null || _monitor$namespaces === void 0 ? void 0 : _monitor$namespaces[0];
      if (!spaceId) {
        continue;
      }
      spaceIds.add(spaceId);
      const normalizedMonitor = (0, _utils.normalizeSecrets)(monitor).attributes;
      const {
        str: paramsString
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], normalizedMonitor);
      if (!configsBySpaces[spaceId]) {
        configsBySpaces[spaceId] = [];
      }
      configsBySpaces[spaceId].push((0, _format_configs.formatHeartbeatRequest)({
        spaceId,
        monitor: normalizedMonitor,
        configId: monitor.id
      }, paramsString));
    }
    return {
      configsBySpaces,
      spaceIds
    };
  }
  async hasAnyParamChanged({
    soClient,
    lastStartedAt,
    lastTotalParams
  }) {
    const {
      logger
    } = this.serverSetup;
    const [editedParams, totalParams] = await Promise.all([soClient.find({
      type: _saved_objects.syntheticsParamType,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      filter: `synthetics-param.updated_at > "${lastStartedAt}"`,
      fields: []
    }), soClient.find({
      type: _saved_objects.syntheticsParamType,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      fields: []
    })]);
    logger.debug(`Found ${editedParams.total} params updated and ${totalParams.total} total params`);
    const updatedParams = editedParams.total;
    const noOfParams = totalParams.total;
    const hasParamsChanges = updatedParams > 0 || noOfParams !== lastTotalParams;
    return {
      hasParamsChanges,
      updatedParams: editedParams.total,
      totalParams: noOfParams
    };
  }
  async hasMWsChanged({
    soClient,
    lastStartedAt,
    lastTotalMWs
  }) {
    const {
      logger
    } = this.serverSetup;
    const [editedMWs, totalMWs] = await Promise.all([soClient.find({
      type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      filter: `${_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE}.updated_at > "${lastStartedAt}"`,
      fields: []
    }), soClient.find({
      type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      fields: []
    })]);
    logger.debug(`Found ${editedMWs.total} maintenance windows updated and ${totalMWs.total} total maintenance windows`);
    const updatedMWs = editedMWs.total;
    const noOfMWs = totalMWs.total;
    const hasMWsChanged = updatedMWs > 0 || noOfMWs !== lastTotalMWs;
    return {
      hasMWsChanged,
      updatedMWs,
      totalMWs: noOfMWs
    };
  }
}
exports.SyncPrivateLocationMonitorsTask = SyncPrivateLocationMonitorsTask;
const runSynPrivateLocationMonitorsTaskSoon = async ({
  server,
  retries = 5
}) => {
  try {
    await (0, _pRetry.default)(async () => {
      const {
        logger,
        pluginsStart: {
          taskManager
        }
      } = server;
      logger.debug(`Scheduling Synthetics sync private location monitors task soon`);
      await taskManager.runSoon(TASK_ID);
      logger.debug(`Synthetics sync private location task scheduled successfully`);
    }, {
      retries
    });
  } catch (error) {
    server.logger.error(`Error scheduling Synthetics sync private location monitors task: ${error.message}`, {
      error
    });
  }
};
exports.runSynPrivateLocationMonitorsTaskSoon = runSynPrivateLocationMonitorsTaskSoon;