"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ServiceAPIClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _axios = _interopRequireDefault(require("axios"));
var _rxjs = require("rxjs");
var https = _interopRequireWildcard(require("https"));
var _serverHttpTools = require("@kbn/server-http-tools");
var _convert_to_data_stream = require("./formatters/public_formatters/convert_to_data_stream");
var _monitor_upgrade_sender = require("../routes/telemetry/monitor_upgrade_sender");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TEST_SERVICE_USERNAME = 'localKibanaIntegrationTestsUser';
class ServiceAPIClient {
  constructor(logger, config, server) {
    (0, _defineProperty2.default)(this, "username", void 0);
    (0, _defineProperty2.default)(this, "authorization", void 0);
    (0, _defineProperty2.default)(this, "locations", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "stackVersion", void 0);
    (0, _defineProperty2.default)(this, "server", void 0);
    this.config = config;
    const {
      username,
      password
    } = config !== null && config !== void 0 ? config : {};
    this.username = username;
    this.stackVersion = server.stackVersion;
    if (username && password) {
      this.authorization = 'Basic ' + Buffer.from(`${username}:${password}`).toString('base64');
    } else {
      this.authorization = '';
    }
    this.logger = logger;
    this.locations = [];
    this.server = server;
  }
  addVersionHeader(req) {
    req.headers = {
      ...req.headers,
      'x-kibana-version': this.stackVersion
    };
    return req;
  }
  async checkAccountAccessStatus() {
    var _this$config;
    if (this.authorization || !((_this$config = this.config) !== null && _this$config !== void 0 && _this$config.manifestUrl)) {
      // in case username/password is provided, we assume it's always allowed
      return {
        allowed: true,
        signupUrl: null
      };
    }
    if (this.locations.length > 0) {
      // get a url from a random location
      const url = this.locations[Math.floor(Math.random() * this.locations.length)].url;

      /* url is required for service locations, but omitted for private locations.
      /* this.locations is only service locations */
      const httpsAgent = this.getHttpsAgent(url);
      if (httpsAgent) {
        try {
          const {
            data
          } = await (0, _axios.default)(this.addVersionHeader({
            method: 'GET',
            url: url + '/allowed',
            httpsAgent
          }));
          const {
            allowed,
            signupUrl
          } = data;
          return {
            allowed,
            signupUrl
          };
        } catch (error) {
          this.logger.error(`Error getting isAllowed status, Error: ${error.message}`, {
            error
          });
        }
      }
    } else {
      this.logger.debug('Failed to fetch isAllowed status. Locations were not fetched from manifest.');
    }
    return {
      allowed: false,
      signupUrl: null
    };
  }
  getHttpsAgent(targetUrl) {
    var _this$config2;
    const parsedTargetUrl = new URL(targetUrl);
    const rejectUnauthorized = parsedTargetUrl.hostname !== 'localhost' || !this.server.isDev;
    const baseHttpsAgent = new https.Agent({
      rejectUnauthorized
    });
    const config = (_this$config2 = this.config) !== null && _this$config2 !== void 0 ? _this$config2 : {};

    // If using basic-auth, ignore certificate configs
    if (this.authorization) return baseHttpsAgent;
    if (config.tls && config.tls.certificate && config.tls.key) {
      const tlsConfig = new _serverHttpTools.SslConfig(config.tls);
      return new https.Agent({
        rejectUnauthorized,
        cert: tlsConfig.certificate,
        key: tlsConfig.key
      });
    } else if (!this.server.isDev) {
      this.logger.warn('TLS certificate and key are not provided. Falling back to default HTTPS agent.');
    }
    return baseHttpsAgent;
  }
  async inspect(data) {
    const monitorsByLocation = this.processServiceData(data);
    return monitorsByLocation.map(({
      data: payload
    }) => payload);
  }
  async post(data) {
    return (await this.callAPI('POST', data)).pushErrors;
  }
  async put(data) {
    return (await this.callAPI('PUT', data)).pushErrors;
  }
  async delete(data) {
    return (await this.callAPI('DELETE', data)).pushErrors;
  }
  async runOnce(data) {
    return (await this.callAPI('POST', {
      ...data,
      endpoint: 'runOnce'
    })).pushErrors;
  }
  async syncMonitors(data) {
    try {
      return (await this.callAPI('PUT', {
        ...data,
        endpoint: 'sync'
      })).pushErrors;
    } catch (error) {
      this.logger.error(`Error syncing Synthetics monitors, Error: ${error.message}`, {
        error
      });
    }
  }
  processServiceData({
    monitors,
    location,
    ...restOfData
  }) {
    // group monitors by location
    const monitorsByLocation = [];
    this.locations.forEach(({
      id,
      url
    }) => {
      if (!location || location.id === id) {
        const locMonitors = monitors.filter(({
          locations
        }) => locations === null || locations === void 0 ? void 0 : locations.find(loc => loc.id === id && loc.isServiceManaged));
        if (locMonitors.length > 0) {
          const data = this.getRequestData({
            ...restOfData,
            monitors: locMonitors
          });
          monitorsByLocation.push({
            location: {
              id,
              url
            },
            monitors: locMonitors,
            data
          });
        }
      }
    });
    return monitorsByLocation;
  }
  async callAPI(method, serviceData) {
    const {
      endpoint
    } = serviceData;
    if (this.username === TEST_SERVICE_USERNAME) {
      // we don't want to call service while local integration tests are running
      return {
        result: [],
        pushErrors: []
      };
    }
    const pushErrors = [];
    const promises = [];
    const monitorsByLocation = this.processServiceData(serviceData);
    monitorsByLocation.forEach(({
      location: {
        url,
        id
      },
      data
    }) => {
      const sendRequest = payload => {
        const promise = this.callServiceEndpoint(payload, method, url, endpoint);
        return (0, _rxjs.from)(promise).pipe((0, _rxjs.tap)(result => {
          this.logSuccessMessage(url, method, payload.monitors.length, result);
        }), (0, _rxjs.catchError)(err => {
          var _err$response, _err$response2;
          if (((_err$response = err.response) === null || _err$response === void 0 ? void 0 : _err$response.status) === 413 && payload.monitors.length > 1) {
            // If payload is too large, split it and retry
            const mid = Math.ceil(payload.monitors.length / 2);
            const firstHalfMonitors = payload.monitors.slice(0, mid);
            const secondHalfMonitors = payload.monitors.slice(mid);
            this.logger.debug(`Payload of ${payload.monitors.length} monitors is too large for location ${id}, splitting in half, in chunks of ${mid}`);
            return (0, _rxjs.concat)(sendRequest({
              ...payload,
              monitors: firstHalfMonitors
            }),
            // Retry with the first half
            sendRequest({
              ...payload,
              monitors: secondHalfMonitors
            }) // Retry with the second half
            );
          }
          pushErrors.push({
            locationId: id,
            error: (_err$response2 = err.response) === null || _err$response2 === void 0 ? void 0 : _err$response2.data
          });
          this.logServiceError(err, url, method, payload.monitors.length);

          // Return an empty observable to prevent unhandled exceptions
          return (0, _rxjs.of)(true);
        }));
      };

      // Start with the initial data payload
      promises.push(sendRequest(data));
    });
    const result = await (0, _rxjs.forkJoin)(promises).toPromise();
    return {
      pushErrors,
      result
    };
  }
  async callServiceEndpoint(data,
  // INSPECT is a special case where we don't want to call the service, but just return the data
  method, baseUrl, endpoint = 'monitors') {
    let url = baseUrl;
    switch (endpoint) {
      case 'monitors':
        url += '/monitors';
        break;
      case 'runOnce':
        url += '/run';
        break;
      case 'sync':
        url += '/monitors/sync';
        break;
    }
    const authHeader = this.authorization ? {
      Authorization: this.authorization
    } : undefined;
    return (0, _axios.default)(this.addVersionHeader({
      method,
      url,
      data,
      headers: authHeader,
      httpsAgent: this.getHttpsAgent(baseUrl)
    }));
  }
  getRequestData({
    monitors,
    output,
    isEdit,
    license
  }) {
    var _this$server$cloud, _this$server$cloud2;
    // don't need to pass locations to heartbeat
    const monitorsStreams = monitors.map(({
      locations,
      ...rest
    }) => (0, _convert_to_data_stream.convertToDataStreamFormat)(rest));
    return {
      monitors: monitorsStreams,
      output,
      stack_version: this.stackVersion,
      is_edit: isEdit,
      license_level: license.type,
      license_issued_to: license.issued_to,
      deployment_id: (_this$server$cloud = this.server.cloud) === null || _this$server$cloud === void 0 ? void 0 : _this$server$cloud.deploymentId,
      cloud_id: (_this$server$cloud2 = this.server.cloud) === null || _this$server$cloud2 === void 0 ? void 0 : _this$server$cloud2.cloudId
    };
  }
  isLoggable(result) {
    const objCast = result;
    return Object.keys(objCast).some(k => k === 'status' || k === 'request');
  }
  logSuccessMessage(url, method, numMonitors, result) {
    if (this.isLoggable(result)) {
      var _result$request;
      if (result.data) {
        this.logger.debug(result.data);
      }
      this.logger.debug(`Successfully called service location ${url}${(_result$request = result.request) === null || _result$request === void 0 ? void 0 : _result$request.path} with method ${method} with ${numMonitors} monitors`);
    }
  }
  logServiceError(err, url, method, numMonitors) {
    var _err$response$data$re, _err$response3, _err$response3$data, _err$request, _err$response4, _err$response4$data, _err$response5, _err$response5$data;
    const reason = (_err$response$data$re = (_err$response3 = err.response) === null || _err$response3 === void 0 ? void 0 : (_err$response3$data = _err$response3.data) === null || _err$response3$data === void 0 ? void 0 : _err$response3$data.reason) !== null && _err$response$data$re !== void 0 ? _err$response$data$re : '';
    err.message = `Failed to call service location ${url}${(_err$request = err.request) === null || _err$request === void 0 ? void 0 : _err$request.path} with method ${method} with ${numMonitors} monitors:  ${err.message}, ${reason}`;
    this.logger.error(err);
    (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(this.logger, this.server.telemetry, {
      reason: (_err$response4 = err.response) === null || _err$response4 === void 0 ? void 0 : (_err$response4$data = _err$response4.data) === null || _err$response4$data === void 0 ? void 0 : _err$response4$data.reason,
      message: err.message,
      type: 'syncError',
      code: err.code,
      status: (_err$response5 = err.response) === null || _err$response5 === void 0 ? void 0 : (_err$response5$data = _err$response5.data) === null || _err$response5$data === void 0 ? void 0 : _err$response5$data.status,
      url,
      stackVersion: this.server.stackVersion
    });
  }
}
exports.ServiceAPIClient = ServiceAPIClient;