"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addPrivateLocationRoute = exports.PrivateLocationSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _uuid = require("uuid");
var _private_location_repository = require("../../../repositories/private_location_repository");
var _feature = require("../../../feature");
var _migrate_legacy_private_locations = require("./migrate_legacy_private_locations");
var _constants = require("../../../../common/constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PrivateLocationSchema = exports.PrivateLocationSchema = _configSchema.schema.object({
  label: _configSchema.schema.string(),
  agentPolicyId: _configSchema.schema.string(),
  tags: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
  geo: _configSchema.schema.maybe(_configSchema.schema.object({
    lat: _configSchema.schema.number(),
    lon: _configSchema.schema.number()
  })),
  spaces: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
    minSize: 1
  }))
});
const addPrivateLocationRoute = () => ({
  method: 'POST',
  path: _constants.SYNTHETICS_API_URLS.PRIVATE_LOCATIONS,
  validate: {},
  validation: {
    request: {
      body: PrivateLocationSchema
    }
  },
  requiredPrivileges: [_feature.PRIVATE_LOCATION_WRITE_API],
  handler: async routeContext => {
    const {
      response,
      request,
      server
    } = routeContext;
    const internalSOClient = server.coreStart.savedObjects.createInternalRepository();
    await (0, _migrate_legacy_private_locations.migrateLegacyPrivateLocations)(internalSOClient, server.logger);
    const repo = new _private_location_repository.PrivateLocationRepository(routeContext);
    const invalidError = await repo.validatePrivateLocation();
    if (invalidError) {
      return invalidError;
    }
    const location = request.body;
    const newId = (0, _uuid.v4)();
    const formattedLocation = (0, _helpers.toSavedObjectContract)({
      ...location,
      id: newId
    });
    const {
      spaces
    } = location;
    try {
      const result = await repo.createPrivateLocation(formattedLocation, newId);
      return (0, _helpers.toClientContract)(result);
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isForbiddenError(error)) {
        if (spaces !== null && spaces !== void 0 && spaces.includes('*')) {
          return response.badRequest({
            body: {
              message: `You do not have permission to create a location in all spaces.`
            }
          });
        }
        return response.customError({
          statusCode: error.output.statusCode,
          body: {
            message: error.message
          }
        });
      }
      throw error;
    }
  }
});
exports.addPrivateLocationRoute = addPrivateLocationRoute;