"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TLSRuleExecutor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _synthetics_settings = require("../../saved_objects/synthetics_settings");
var _saved_objects = require("../../../common/types/saved_objects");
var _monitor_config_repository = require("../../services/monitor_config_repository");
var _client_defaults = require("../../../common/constants/client_defaults");
var _common = require("../common");
var _get_certs = require("../../queries/get_certs");
var _constants = require("../../../common/constants");
var _process_monitors = require("../../saved_objects/synthetics_monitor/process_monitors");
var _runtime_types = require("../../../common/runtime_types");
var _monitor_management = require("../../../common/constants/monitor_management");
var _lib = require("../../lib");
var _filter_monitors = require("../status_rule/queries/filter_monitors");
var _common2 = require("../../routes/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class TLSRuleExecutor {
  constructor(previousStartedAt, p, soClient, scopedClient, server, syntheticsMonitorClient, spaceId, ruleName) {
    (0, _defineProperty2.default)(this, "previousStartedAt", void 0);
    (0, _defineProperty2.default)(this, "params", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "syntheticsMonitorClient", void 0);
    (0, _defineProperty2.default)(this, "monitors", []);
    (0, _defineProperty2.default)(this, "monitorConfigRepository", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    (0, _defineProperty2.default)(this, "ruleName", void 0);
    (0, _defineProperty2.default)(this, "getRuleThresholdOverview", async () => {
      await this.getMonitors();
      return {
        monitors: this.monitors.map(monitor => ({
          id: monitor.id,
          name: monitor.attributes.name,
          type: monitor.attributes.type
        }))
      }; // The returned object is cast to TLSRuleInspect because the AlertOverviewStatus is not included. The AlertOverviewStatus is probably not used in the frontend, we should check if it is still needed
    });
    this.previousStartedAt = previousStartedAt;
    this.params = p;
    this.soClient = soClient;
    this.esClient = new _lib.SyntheticsEsClient(this.soClient, scopedClient, {
      heartbeatIndices: _constants.SYNTHETICS_INDEX_PATTERN
    });
    this.server = server;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
    this.monitorConfigRepository = new _monitor_config_repository.MonitorConfigRepository(soClient, server.encryptedSavedObjects.getClient());
    this.logger = server.logger;
    this.spaceId = spaceId;
    this.ruleName = ruleName;
  }
  debug(message) {
    this.logger.debug(`[TLS Rule Executor][${this.ruleName}] ${message}`);
  }
  async getMonitors() {
    var _this$params, _this$params2, _this$params3, _this$params4;
    const HTTP_OR_TCP = `${_saved_objects.syntheticsMonitorAttributes}.${_runtime_types.ConfigKey.MONITOR_TYPE}: http or ${_saved_objects.syntheticsMonitorAttributes}.${_runtime_types.ConfigKey.MONITOR_TYPE}: tcp`;
    const baseFilter = `${_saved_objects.syntheticsMonitorAttributes}.${_monitor_management.AlertConfigKey.TLS_ENABLED}: true and (${HTTP_OR_TCP})`;
    const configIds = await (0, _filter_monitors.queryFilterMonitors)({
      spaceId: this.spaceId,
      esClient: this.esClient,
      ruleParams: this.params
    });
    if (this.params.kqlQuery && (0, _lodash.isEmpty)(configIds)) {
      this.debug(`No monitor found with the given KQL query ${this.params.kqlQuery}`);
      return (0, _process_monitors.processMonitors)([]);
    }
    const locationIds = await (0, _common2.parseLocationFilter)({
      savedObjectsClient: this.soClient,
      server: this.server,
      syntheticsMonitorClient: this.syntheticsMonitorClient
    }, this.params.locations);
    const {
      filtersStr
    } = (0, _common2.parseArrayFilters)({
      configIds,
      filter: baseFilter,
      tags: (_this$params = this.params) === null || _this$params === void 0 ? void 0 : _this$params.tags,
      locations: locationIds,
      monitorTypes: (_this$params2 = this.params) === null || _this$params2 === void 0 ? void 0 : _this$params2.monitorTypes,
      monitorQueryIds: (_this$params3 = this.params) === null || _this$params3 === void 0 ? void 0 : _this$params3.monitorIds,
      projects: (_this$params4 = this.params) === null || _this$params4 === void 0 ? void 0 : _this$params4.projects
    });
    this.monitors = await this.monitorConfigRepository.getAll({
      filter: filtersStr
    });
    this.debug(`Found ${this.monitors.length} monitors for params ${JSON.stringify(this.params)} | parsed location filter is ${JSON.stringify(locationIds)} `);
    const {
      enabledMonitorQueryIds
    } = (0, _process_monitors.processMonitors)(this.monitors);
    return {
      enabledMonitorQueryIds
    };
  }
  async getExpiredCertificates() {
    var _ref, _this$params$certExpi, _ref2, _this$params$certAgeT;
    const {
      enabledMonitorQueryIds
    } = await this.getMonitors();
    const dynamicSettings = await (0, _synthetics_settings.getSyntheticsDynamicSettings)(this.soClient);
    const expiryThreshold = (_ref = (_this$params$certExpi = this.params.certExpirationThreshold) !== null && _this$params$certExpi !== void 0 ? _this$params$certExpi : dynamicSettings === null || dynamicSettings === void 0 ? void 0 : dynamicSettings.certExpirationThreshold) !== null && _ref !== void 0 ? _ref : _constants.DYNAMIC_SETTINGS_DEFAULTS.certExpirationThreshold;
    const ageThreshold = (_ref2 = (_this$params$certAgeT = this.params.certAgeThreshold) !== null && _this$params$certAgeT !== void 0 ? _this$params$certAgeT : dynamicSettings === null || dynamicSettings === void 0 ? void 0 : dynamicSettings.certAgeThreshold) !== null && _ref2 !== void 0 ? _ref2 : _constants.DYNAMIC_SETTINGS_DEFAULTS.certAgeThreshold;
    const absoluteExpirationThreshold = (0, _moment.default)().add(expiryThreshold, 'd').valueOf();
    const absoluteAgeThreshold = (0, _moment.default)().subtract(ageThreshold, 'd').valueOf();
    if (enabledMonitorQueryIds.length === 0) {
      return {
        latestPings: [],
        certs: [],
        total: 0,
        foundCerts: false,
        expiryThreshold,
        ageThreshold,
        absoluteExpirationThreshold,
        absoluteAgeThreshold
      };
    }
    let filters;
    if (this.params.search) {
      filters = await (0, _common.formatFilterString)(this.esClient, undefined, this.params.search);
    }
    const {
      certs,
      total
    } = await (0, _get_certs.getSyntheticsCerts)({
      syntheticsEsClient: this.esClient,
      pageIndex: 0,
      size: 1000,
      notValidAfter: `now+${expiryThreshold}d`,
      notValidBefore: `now-${ageThreshold}d`,
      sortBy: 'common_name',
      direction: 'desc',
      filters,
      monitorIds: enabledMonitorQueryIds
    });
    this.debug(`Found ${certs.length} certificates: ` + certs.map(cert => cert.sha256).join(', '));
    const latestPings = await this.getLatestPingsForMonitors(certs);
    const foundCerts = total > 0;
    return {
      latestPings,
      foundCerts,
      total,
      expiryThreshold,
      ageThreshold,
      absoluteExpirationThreshold,
      absoluteAgeThreshold,
      certs: this.filterOutResolvedCerts(certs, latestPings)
    };
  }
  filterOutResolvedCerts(certs, latestPings) {
    const latestPingsMap = new Map();
    latestPings.forEach(ping => {
      latestPingsMap.set(ping.config_id, ping);
    });
    return certs.filter(cert => {
      const lPing = latestPingsMap.get(cert.configId);
      if (!lPing) {
        return true;
      }
      return (0, _moment.default)(lPing['@timestamp']).isBefore(cert['@timestamp']);
    });
  }
  async getLatestPingsForMonitors(certs) {
    if (certs.length === 0) {
      return [];
    }
    const configIds = certs.map(cert => cert.configId);
    const certIds = certs.map(cert => cert.sha256);
    const {
      body
    } = await this.esClient.search({
      body: {
        query: {
          bool: {
            filter: [{
              range: {
                '@timestamp': {
                  gte: 'now-1d',
                  lt: 'now'
                }
              }
            }, {
              terms: {
                config_id: configIds
              }
            }, _client_defaults.FINAL_SUMMARY_FILTER],
            must_not: {
              bool: {
                filter: [{
                  terms: {
                    'tls.server.hash.sha256': certIds
                  }
                }]
              }
            }
          }
        },
        collapse: {
          field: 'config_id'
        },
        _source: ['@timestamp', 'monitor', 'url', 'config_id', 'tls'],
        sort: [{
          '@timestamp': {
            order: 'desc'
          }
        }]
      }
    });
    return body.hits.hits.map(hit => hit._source);
  }
}
exports.TLSRuleExecutor = TLSRuleExecutor;