"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.internalLifecycleRoutes = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _zod = require("@kbn/zod");
var _constants = require("../../../../../common/constants");
var _create_server_route = require("../../../create_server_route");
var _ilm_phases = require("../../../../lib/streams/lifecycle/ilm_phases");
var _get_effective_lifecycle = require("../../../../lib/streams/lifecycle/get_effective_lifecycle");
var _status_error = require("../../../../lib/streams/errors/status_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const lifecycleStatsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/lifecycle/_stats',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const name = params.path.name;
    const definition = await streamsClient.getStream(name);
    if (!_streamsSchema.Streams.ingest.all.Definition.is(definition)) {
      throw new _status_error.StatusError('Lifecycle stats are only available for ingest streams', 400);
    }
    const dataStream = await streamsClient.getDataStream(name);
    const lifecycle = await (0, _get_effective_lifecycle.getEffectiveLifecycle)({
      definition,
      streamsClient,
      dataStream
    });
    if (!(0, _streamsSchema.isIlmLifecycle)(lifecycle)) {
      throw new _status_error.StatusError('Lifecycle stats are only available for ILM policy', 400);
    }
    const {
      policy
    } = await scopedClusterClient.asCurrentUser.ilm.getLifecycle({
      name: lifecycle.ilm.policy
    }).then(policies => policies[lifecycle.ilm.policy]);
    const [{
      indices: indicesIlmDetails
    }, {
      indices: indicesStats = {}
    }] = await Promise.all([scopedClusterClient.asCurrentUser.ilm.explainLifecycle({
      index: name
    }), scopedClusterClient.asCurrentUser.indices.stats({
      index: dataStream.name
    })]);
    return {
      phases: (0, _ilm_phases.ilmPhases)({
        policy,
        indicesIlmDetails,
        indicesStats
      })
    };
  }
});
const lifecycleIlmExplainRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/lifecycle/_explain',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const name = params.path.name;

    // verifies read privileges
    await streamsClient.getStream(name);
    return scopedClusterClient.asCurrentUser.ilm.explainLifecycle({
      index: name
    });
  }
});
const internalLifecycleRoutes = exports.internalLifecycleRoutes = {
  ...lifecycleStatsRoute,
  ...lifecycleIlmExplainRoute
};