"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIngestionRatePerTier = exports.useIngestionRate = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/data-plugin/common");
var _rxjs = require("rxjs");
var _use_kibana = require("../../../../hooks/use_kibana");
var _use_streams_app_fetch = require("../../../../hooks/use_streams_app_fetch");
var _use_ilm_phases_color_and_description = require("./use_ilm_phases_color_and_description");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TIMESTAMP_FIELD = '@timestamp';
const RANDOM_SAMPLER_PROBABILITY = 0.1;

// some units are not supported for the fixed_interval of the date histogram
// this function uses the calculateAutoTimeExpression function to determine
// if the interval should be a calendar_interval or a fixed_interval
const getIntervalAndType = (timeState, core) => {
  const interval = (0, _common.getCalculateAutoTimeExpression)(key => core.uiSettings.get(key))(timeState.asAbsoluteTimeRange);
  if (!interval) {
    return undefined;
  }
  const calendarIntervalUnits = new Set(['w', 'M', 'q', 'y']);
  const intervalType = calendarIntervalUnits.has(interval.replace(/^\d+/, '')) ? 'calendar_interval' : 'fixed_interval';
  return {
    interval,
    intervalType
  };
};
const useIngestionRate = ({
  definition,
  stats,
  timeState
}) => {
  const {
    core,
    dependencies: {
      start: {
        data
      }
    }
  } = (0, _use_kibana.useKibana)();
  const ingestionRateFetch = (0, _use_streams_app_fetch.useStreamsAppFetch)(async ({
    signal
  }) => {
    if (!stats) {
      return;
    }
    const intervalData = getIntervalAndType(timeState, core);
    if (!intervalData) {
      return;
    }
    const {
      interval,
      intervalType
    } = intervalData;
    const {
      rawResponse: {
        aggregations
      }
    } = await (0, _rxjs.lastValueFrom)(data.search.search({
      params: {
        index: definition.stream.name,
        track_total_hits: false,
        body: {
          size: 0,
          query: {
            bool: {
              filter: [{
                range: {
                  [TIMESTAMP_FIELD]: {
                    gte: timeState.start,
                    lte: timeState.end
                  }
                }
              }]
            }
          },
          aggs: {
            sampler: {
              random_sampler: {
                probability: RANDOM_SAMPLER_PROBABILITY,
                seed: 42
              },
              aggs: {
                docs_count: {
                  date_histogram: {
                    field: TIMESTAMP_FIELD,
                    [intervalType]: interval,
                    min_doc_count: 0,
                    extended_bounds: {
                      min: timeState.start,
                      max: timeState.end
                    }
                  }
                }
              }
            }
          }
        }
      }
    }, {
      abortSignal: signal
    }));
    return {
      start: (0, _moment.default)(timeState.start),
      end: (0, _moment.default)(timeState.end),
      interval,
      buckets: aggregations.sampler.docs_count.buckets.map(({
        key,
        doc_count: docCount
      }) => ({
        key,
        value: docCount * stats.bytesPerDoc
      }))
    };
  }, [definition, stats, timeState, core, data.search]);
  return {
    ingestionRate: ingestionRateFetch.value,
    isLoading: ingestionRateFetch.loading,
    error: ingestionRateFetch.error
  };
};
exports.useIngestionRate = useIngestionRate;
const useIngestionRatePerTier = ({
  definition,
  stats,
  timeState
}) => {
  const {
    core,
    dependencies: {
      start: {
        data,
        streams: {
          streamsRepositoryClient
        }
      }
    }
  } = (0, _use_kibana.useKibana)();
  const {
    ilmPhases
  } = (0, _use_ilm_phases_color_and_description.useIlmPhasesColorAndDescription)();
  const ingestionRateFetch = (0, _use_streams_app_fetch.useStreamsAppFetch)(async ({
    signal
  }) => {
    if (!stats) {
      return;
    }
    const intervalData = getIntervalAndType(timeState, core);
    if (!timeState.start || !timeState.end || !intervalData) {
      return;
    }
    const start = (0, _moment.default)(timeState.start);
    const end = (0, _moment.default)(timeState.end);
    const {
      interval,
      intervalType
    } = intervalData;
    const {
      rawResponse: {
        aggregations
      }
    } = await (0, _rxjs.lastValueFrom)(data.search.search({
      params: {
        index: definition.stream.name,
        track_total_hits: false,
        body: {
          size: 0,
          query: {
            bool: {
              filter: [{
                range: {
                  [TIMESTAMP_FIELD]: {
                    gte: start,
                    lte: end
                  }
                }
              }]
            }
          },
          aggs: {
            sampler: {
              random_sampler: {
                probability: RANDOM_SAMPLER_PROBABILITY,
                seed: 42
              },
              aggs: {
                docs_count: {
                  date_histogram: {
                    field: TIMESTAMP_FIELD,
                    [intervalType]: interval,
                    min_doc_count: 0,
                    extended_bounds: {
                      min: start,
                      max: end
                    }
                  },
                  aggs: {
                    indices: {
                      terms: {
                        field: '_index'
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }, {
      abortSignal: signal
    }));
    if (aggregations.sampler.docs_count.buckets.length === 0) {
      return {
        start,
        end,
        interval,
        buckets: {}
      };
    }
    const ilmExplain = await streamsRepositoryClient.fetch('GET /internal/streams/{name}/lifecycle/_explain', {
      params: {
        path: {
          name: definition.stream.name
        }
      },
      signal
    });
    const fallbackTier = 'hot';
    const buckets = aggregations.sampler.docs_count.buckets.reduce((acc, {
      key,
      doc_count: docCount,
      indices
    }) => {
      if (docCount === 0) {
        var _acc$fallbackTier;
        // there's no data for this bucket. push an empty data point
        // so we still graph the timestamp.
        (acc[fallbackTier] = (_acc$fallbackTier = acc[fallbackTier]) !== null && _acc$fallbackTier !== void 0 ? _acc$fallbackTier : []).push({
          key,
          value: 0
        });
        return acc;
      }
      const countByTier = indices.buckets.reduce((tiers, index) => {
        var _tiers$tier;
        const explain = ilmExplain.indices[index.key];
        const tier = explain.managed && explain.phase in ilmPhases ? explain.phase : fallbackTier;
        tiers[tier] = ((_tiers$tier = tiers[tier]) !== null && _tiers$tier !== void 0 ? _tiers$tier : 0) + index.doc_count;
        return tiers;
      }, {});
      for (const entry of Object.entries(countByTier)) {
        var _acc$tier;
        const tier = entry[0];
        (acc[tier] = (_acc$tier = acc[tier]) !== null && _acc$tier !== void 0 ? _acc$tier : []).push({
          key,
          value: entry[1] * stats.bytesPerDoc
        });
      }
      return acc;
    }, {});
    return {
      start,
      end,
      interval,
      buckets
    };
  }, [definition, stats, timeState, core, data.search, streamsRepositoryClient, ilmPhases]);
  return {
    ingestionRate: ingestionRateFetch.value,
    isLoading: ingestionRateFetch.loading,
    error: ingestionRateFetch.error
  };
};
exports.useIngestionRatePerTier = useIngestionRatePerTier;