"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrowdstrikeConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _rtr_session_manager = require("./rtr_session_manager");
var _types = require("./types");
var _schema = require("../../../common/crowdstrike/schema");
var _constants = require("../../../common/crowdstrike/constants");
var _error = require("./error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SUPPORTED_RTR_COMMANDS = ['runscript'];
const paramsSerializer = params => {
  return Object.entries(params).map(([key, value]) => `${encodeURIComponent(key)}=${encodeURIComponent(value)}`).join('&');
};

/**
 * Crowdstrike Connector
 * @constructor
 * @param {string} token - Authorization token received from OAuth2 API, that needs to be sent along with each request.
 * @param {number} tokenExpiryTimeout - Tokens are valid for 30 minutes, so we will refresh them every 29 minutes
 * @param {base64} base64encodedToken - The base64 encoded token used for authentication.
 */

class CrowdstrikeConnector extends _server.SubActionConnector {
  constructor(params, experimentalFeatures) {
    super(params);
    (0, _defineProperty2.default)(this, "experimentalFeatures", void 0);
    (0, _defineProperty2.default)(this, "crowdStrikeSessionManager", void 0);
    (0, _defineProperty2.default)(this, "urls", void 0);
    (0, _defineProperty2.default)(this, "getTokenRequest", async connectorUsageCollector => {
      var _response$data;
      const response = await this.request({
        url: this.urls.getToken,
        method: 'post',
        headers: {
          accept: 'application/json',
          'Content-Type': 'application/x-www-form-urlencoded',
          authorization: 'Basic ' + CrowdstrikeConnector.base64encodedToken
        },
        responseSchema: _schema.CrowdstrikeApiDoNotValidateResponsesSchema
      }, connectorUsageCollector);
      const token = (_response$data = response.data) === null || _response$data === void 0 ? void 0 : _response$data.access_token;
      if (token) {
        // Clear any existing timeout
        clearTimeout(CrowdstrikeConnector.tokenExpiryTimeout);

        // Set a timeout to reset the token after 29 minutes (it expires after 30 minutes)
        CrowdstrikeConnector.tokenExpiryTimeout = setTimeout(() => {
          CrowdstrikeConnector.token = null;
        }, 29 * 60 * 1000);
      }
      return token;
    });
    (0, _defineProperty2.default)(this, "crowdstrikeApiRequest", async (req, connectorUsageCollector, retried) => {
      try {
        if (!CrowdstrikeConnector.token) {
          CrowdstrikeConnector.token = await this.getTokenRequest(connectorUsageCollector);
        }
        const response = await this.request({
          ...req,
          // We don't validate responses from Crowdstrike API's because we do not want failures for cases
          // where the external system might add/remove/change values in the response that we have no
          // control over.
          responseSchema: _schema.CrowdstrikeApiDoNotValidateResponsesSchema,
          headers: {
            ...req.headers,
            Authorization: `Bearer ${CrowdstrikeConnector.token}`
          }
        }, connectorUsageCollector);
        return response.data;
      } catch (error) {
        if (error.code === 401 && !retried) {
          CrowdstrikeConnector.token = null;
          return this.crowdstrikeApiRequest(req, connectorUsageCollector, true);
        }
        throw new _error.CrowdstrikeError(error.message);
      }
    });
    // Helper method to execute RTR commands with different API endpoints
    (0, _defineProperty2.default)(this, "executeRTRCommandWithUrl", async (url, payload, connectorUsageCollector) => {
      const batchId = await this.crowdStrikeSessionManager.initializeSession({
        endpoint_ids: payload.endpoint_ids
      }, connectorUsageCollector);
      const baseCommand = payload.command.split(' ')[0];
      if (!SUPPORTED_RTR_COMMANDS.includes(baseCommand)) {
        throw new _error.CrowdstrikeError('Command not supported');
      }
      return await this.crowdstrikeApiRequest({
        url,
        method: 'post',
        data: {
          base_command: baseCommand,
          command_string: payload.command,
          batch_id: batchId,
          hosts: payload.endpoint_ids,
          persist_all: false
        },
        paramsSerializer,
        responseSchema: _schema.CrowdstrikeExecuteRTRResponseSchema
      }, connectorUsageCollector);
    });
    this.experimentalFeatures = experimentalFeatures;
    this.urls = {
      getToken: `${this.config.url}/oauth2/token`,
      hostAction: `${this.config.url}/devices/entities/devices-actions/v2`,
      agents: `${this.config.url}/devices/entities/devices/v2`,
      agentStatus: `${this.config.url}/devices/entities/online-state/v1`,
      batchInitRTRSession: `${this.config.url}/real-time-response/combined/batch-init-session/v1`,
      batchRefreshRTRSession: `${this.config.url}/real-time-response/combined/batch-refresh-session/v1`,
      batchExecuteRTR: `${this.config.url}/real-time-response/combined/batch-command/v1`,
      batchActiveResponderExecuteRTR: `${this.config.url}/real-time-response/combined/batch-active-responder-command/v1`,
      batchAdminExecuteRTR: `${this.config.url}/real-time-response/combined/batch-admin-command/v1`,
      getRTRCloudScripts: `${this.config.url}/real-time-response/entities/scripts/v1`
    };
    if (!CrowdstrikeConnector.base64encodedToken) {
      CrowdstrikeConnector.base64encodedToken = Buffer.from(this.secrets.clientId + ':' + this.secrets.clientSecret).toString('base64');
    }
    this.crowdStrikeSessionManager = new _rtr_session_manager.CrowdStrikeSessionManager(this.urls, this.crowdstrikeApiRequest);
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.SUB_ACTION.GET_AGENT_DETAILS,
      method: 'getAgentDetails',
      schema: _schema.CrowdstrikeGetAgentsParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.HOST_ACTIONS,
      method: 'executeHostActions',
      schema: _schema.CrowdstrikeHostActionsParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.GET_AGENT_ONLINE_STATUS,
      method: 'getAgentOnlineStatus',
      schema: _schema.CrowdstrikeGetAgentsParamsSchema
    });
    if (this.experimentalFeatures.crowdstrikeConnectorRTROn) {
      this.registerSubAction({
        name: _constants.SUB_ACTION.EXECUTE_RTR_COMMAND,
        method: 'executeRTRCommand',
        schema: _schema.CrowdstrikeRTRCommandParamsSchema // Define a proper schema for the command
      });
      this.registerSubAction({
        name: _constants.SUB_ACTION.EXECUTE_ACTIVE_RESPONDER_RTR,
        method: 'batchActiveResponderExecuteRTR',
        schema: _schema.CrowdstrikeRTRCommandParamsSchema // Define a proper schema for the command
      });
      this.registerSubAction({
        name: _constants.SUB_ACTION.EXECUTE_ADMIN_RTR,
        method: 'batchAdminExecuteRTR',
        schema: _schema.CrowdstrikeRTRCommandParamsSchema // Define a proper schema for the command
      });
      this.registerSubAction({
        name: _constants.SUB_ACTION.GET_RTR_CLOUD_SCRIPTS,
        method: 'getRTRCloudScripts',
        schema: _schema.CrowdstrikeRTRCommandParamsSchema // Empty schema - this request do not have any parameters
      });
    }
  }
  async executeHostActions({
    alertIds,
    ...payload
  }, connectorUsageCollector) {
    return this.crowdstrikeApiRequest({
      url: this.urls.hostAction,
      method: 'post',
      params: {
        action_name: payload.command
      },
      data: {
        ids: payload.ids,
        ...(payload.actionParameters ? {
          action_parameters: Object.entries(payload.actionParameters).map(([name, value]) => ({
            name,
            value
          }))
        } : {})
      },
      paramsSerializer,
      responseSchema: _schema.CrowdstrikeHostActionsResponseSchema
    }, connectorUsageCollector);
  }
  async getAgentDetails(payload, connectorUsageCollector) {
    return this.crowdstrikeApiRequest({
      url: this.urls.agents,
      method: 'GET',
      params: {
        ids: payload.ids
      },
      paramsSerializer,
      responseSchema: _schema.RelaxedCrowdstrikeBaseApiResponseSchema
    }, connectorUsageCollector);
  }
  async getAgentOnlineStatus(payload, connectorUsageCollector) {
    return this.crowdstrikeApiRequest({
      url: this.urls.agentStatus,
      method: 'GET',
      params: {
        ids: payload.ids
      },
      paramsSerializer,
      responseSchema: _schema.RelaxedCrowdstrikeBaseApiResponseSchema
    }, connectorUsageCollector);
  }
  // Public method for generic RTR command execution
  async executeRTRCommand(payload, connectorUsageCollector) {
    return await this.executeRTRCommandWithUrl(this.urls.batchExecuteRTR, payload, connectorUsageCollector);
  }

  // Public method for Active Responder RTR command execution
  async batchActiveResponderExecuteRTR(payload, connectorUsageCollector) {
    return await this.executeRTRCommandWithUrl(this.urls.batchActiveResponderExecuteRTR, payload, connectorUsageCollector);
  }

  // Public method for Admin RTR command execution
  async batchAdminExecuteRTR(payload, connectorUsageCollector) {
    return await this.executeRTRCommandWithUrl(this.urls.batchAdminExecuteRTR, payload, connectorUsageCollector);
  }
  async getRTRCloudScripts(payload, connectorUsageCollector) {
    return await this.crowdstrikeApiRequest({
      url: this.urls.getRTRCloudScripts,
      method: 'GET',
      paramsSerializer,
      responseSchema: _schema.CrowdstrikeGetScriptsResponseSchema
    }, connectorUsageCollector);
  }
  getResponseErrorMessage(error) {
    var _error$response, _error$response$data, _error$response$data$, _error$response2, _error$response$data3;
    const errorData = (_error$response = error.response) === null || _error$response === void 0 ? void 0 : (_error$response$data = _error$response.data) === null || _error$response$data === void 0 ? void 0 : (_error$response$data$ = _error$response$data.errors) === null || _error$response$data$ === void 0 ? void 0 : _error$response$data$[0];
    if (errorData) {
      return errorData.message;
    }
    const cause = (0, _types.isAggregateError)(error.cause) ? error.cause.errors[0] : error.cause;
    if (cause) {
      // ENOTFOUND is the error code for when the host is unreachable eg. api.crowdstrike.com111
      if (cause.code === 'ENOTFOUND') {
        return `URL not found: ${cause.hostname}`;
      }
      // ECONNREFUSED is the error code for when the host is unreachable eg. http://MacBook-Pro-Tomasz.local:55555
      if (cause.code === 'ECONNREFUSED') {
        return `Connection Refused: ${cause.address}:${cause.port}`;
      }
    }
    if (!((_error$response2 = error.response) !== null && _error$response2 !== void 0 && _error$response2.status)) {
      var _error$response$data2, _error$response3;
      return `Unknown API Error: ${JSON.stringify((_error$response$data2 = (_error$response3 = error.response) === null || _error$response3 === void 0 ? void 0 : _error$response3.data) !== null && _error$response$data2 !== void 0 ? _error$response$data2 : {})}`;
    }
    return `API Error: ${JSON.stringify((_error$response$data3 = error.response.data) !== null && _error$response$data3 !== void 0 ? _error$response$data3 : {})}`;
  }
}
exports.CrowdstrikeConnector = CrowdstrikeConnector;
(0, _defineProperty2.default)(CrowdstrikeConnector, "token", void 0);
(0, _defineProperty2.default)(CrowdstrikeConnector, "tokenExpiryTimeout", void 0);
(0, _defineProperty2.default)(CrowdstrikeConnector, "base64encodedToken", void 0);