"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.observableIntoEventSourceStream = observableIntoEventSourceStream;
var _errors = require("@kbn/sse-utils/src/errors");
var _events = require("@kbn/sse-utils/src/events");
var _rxjs = require("rxjs");
var _stream = require("stream");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function observableIntoEventSourceStream(source$, {
  logger,
  signal
}) {
  const withSerializedErrors$ = source$.pipe((0, _rxjs.catchError)(error => {
    if ((0, _errors.isSSEError)(error)) {
      logger.error(error);
      logger.debug(() => JSON.stringify(error));
      return (0, _rxjs.of)({
        type: _events.ServerSentEventType.error,
        error: {
          code: error.code,
          message: error.message,
          meta: error.meta
        }
      });
    }
    logger.error(error);
    return (0, _rxjs.of)({
      type: _events.ServerSentEventType.error,
      error: {
        code: _errors.ServerSentEventErrorCode.internalError,
        message: error.message
      }
    });
  }), (0, _rxjs.map)(event => {
    const {
      type,
      ...rest
    } = event;
    return `event: ${type}\ndata: ${JSON.stringify(rest)}\n\n`;
  }));
  const stream = new _stream.PassThrough();
  const intervalId = setInterval(() => {
    // `:` denotes a comment - this is to keep the connection open
    // it will be ignored by the SSE parser on the client
    stream.write(': keep-alive');
  }, 10000);
  const subscription = withSerializedErrors$.subscribe({
    next: line => {
      stream.write(line);
    },
    complete: () => {
      stream.end();
      clearTimeout(intervalId);
    },
    error: error => {
      clearTimeout(intervalId);
      stream.write(`event:error\ndata: ${JSON.stringify({
        error: {
          code: _errors.ServerSentEventErrorCode.internalError,
          message: error.message
        }
      })}\n\n`);
      stream.end();
    }
  });
  signal.addEventListener('abort', () => {
    subscription.unsubscribe();
    stream.end();
  });
  return stream;
}