"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initTour = initTour;
var _rxjs = require("rxjs");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function initTour(core, spacesManager) {
  const showTourUiSettingValue = core.settings.globalClient.get(_constants.SHOW_SPACE_SOLUTION_TOUR_SETTING);
  const showTour$ = new _rxjs.BehaviorSubject(showTourUiSettingValue !== null && showTourUiSettingValue !== void 0 ? showTourUiSettingValue : true);
  const allSpaces$ = (0, _rxjs.defer)(() => (0, _rxjs.from)(spacesManager.getSpaces())).pipe((0, _rxjs.shareReplay)(1));
  const hasMultipleSpaces = spaces => {
    return spaces.length > 1;
  };
  const isDefaultSpaceOnClassic = spaces => {
    const defaultSpace = spaces.find(space => space.id === _constants.DEFAULT_SPACE_ID);
    if (!defaultSpace) {
      // Don't show the tour if the default space doesn't exist (this should never happen)
      return true;
    }
    if (!defaultSpace.solution || defaultSpace.solution === _constants.SOLUTION_VIEW_CLASSIC) {
      return true;
    }
  };
  const showTourObservable$ = showTour$.pipe((0, _rxjs.switchMap)(showTour => {
    if (!showTour) return (0, _rxjs.of)(false);
    return allSpaces$.pipe((0, _rxjs.map)(spaces => {
      if (hasMultipleSpaces(spaces) || isDefaultSpaceOnClassic(spaces)) {
        return false;
      }
      return true;
    }));
  }));
  const hideTourInGlobalSettings = () => {
    core.settings.globalClient.set(_constants.SHOW_SPACE_SOLUTION_TOUR_SETTING, false).catch(() => {
      // Silently swallow errors, the user will just see the tour again next time they load the page
    });
  };
  if (showTourUiSettingValue !== false) {
    allSpaces$.subscribe(spaces => {
      if (hasMultipleSpaces(spaces) || isDefaultSpaceOnClassic(spaces)) {
        // If we have either (1) multiple space or (2) only one space and it's the default space with the classic solution,
        // we don't want to show the tour later on. This can happen in the following scenarios:
        // - the user deletes all the spaces but one (and that last space has a solution set)
        // - the user edits the default space and sets a solution
        // So we can immediately hide the tour in the global settings from now on.
        hideTourInGlobalSettings();
      }
    });
  }
  const onFinishTour = () => {
    hideTourInGlobalSettings();
    showTour$.next(false);
  };
  return {
    showTour$: showTourObservable$,
    onFinishTour
  };
}