"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UpdateSLO = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _std = require("@kbn/std");
var _lodash = require("lodash");
var _constants = require("../../common/constants");
var _sli_pipeline_template = require("../assets/ingest_templates/sli_pipeline_template");
var _summary_pipeline_template = require("../assets/ingest_templates/summary_pipeline_template");
var _services = require("../domain/services");
var _errors = require("../errors");
var _retry = require("../utils/retry");
var _create_temp_summary = require("./summary_transform_generator/helpers/create_temp_summary");
var _assert_expected_indicator_source_index_privileges = require("./utils/assert_expected_indicator_source_index_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class UpdateSLO {
  constructor(repository, transformManager, summaryTransformManager, scopedClusterClient, logger, spaceId, basePath, userId) {
    this.repository = repository;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.scopedClusterClient = scopedClusterClient;
    this.logger = logger;
    this.spaceId = spaceId;
    this.basePath = basePath;
    this.userId = userId;
  }
  async execute(sloId, params) {
    const originalSlo = await this.repository.findById(sloId);
    let updatedSlo = Object.assign({}, originalSlo, {
      ...params,
      groupBy: !!params.groupBy ? params.groupBy : originalSlo.groupBy,
      settings: Object.assign({}, originalSlo.settings, params.settings)
    });
    if ((0, _lodash.isEqual)(originalSlo, updatedSlo)) {
      return this.toResponse(originalSlo);
    }
    const requireRevisionBump = await this.isRevisionBumpRequired(originalSlo, updatedSlo);
    updatedSlo = Object.assign(updatedSlo, {
      updatedAt: new Date(),
      updatedBy: this.userId,
      revision: requireRevisionBump ? originalSlo.revision + 1 : originalSlo.revision
    });
    (0, _services.validateSLO)(updatedSlo);
    await (0, _assert_expected_indicator_source_index_privileges.assertExpectedIndicatorSourceIndexPrivileges)(updatedSlo, this.scopedClusterClient.asCurrentUser);
    const rollbackOperations = [];
    await this.repository.update(updatedSlo);
    rollbackOperations.push(() => this.repository.update(originalSlo));
    if (!requireRevisionBump) {
      // we only have to update the rollup and summary pipelines to include the non-breaking changes (name, desc, tags, ...) in the summary index
      try {
        await this.createPipeline((0, _sli_pipeline_template.getSLIPipelineTemplate)(updatedSlo, this.spaceId));
        rollbackOperations.push(() => this.deletePipeline((0, _constants.getSLOPipelineId)(updatedSlo.id, updatedSlo.revision)));
        await this.createPipeline((0, _summary_pipeline_template.getSummaryPipelineTemplate)(updatedSlo, this.spaceId, this.basePath));
      } catch (err) {
        var _err$meta, _err$meta$body, _err$meta$body$error;
        this.logger.debug(`Cannot update the SLO summary pipeline [id: ${updatedSlo.id}, revision: ${updatedSlo.revision}]. ${err}`);
        await (0, _std.asyncForEach)(rollbackOperations.reverse(), async operation => {
          try {
            await operation();
          } catch (rollbackErr) {
            this.logger.debug(`Rollback operation failed. ${rollbackErr}`);
          }
        });
        if (((_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) === 'security_exception') {
          throw new _errors.SecurityException(err.meta.body.error.reason);
        }
        throw err;
      }
      return this.toResponse(updatedSlo);
    }
    const updatedRollupTransformId = (0, _constants.getSLOTransformId)(updatedSlo.id, updatedSlo.revision);
    const updatedSummaryTransformId = (0, _constants.getSLOSummaryTransformId)(updatedSlo.id, updatedSlo.revision);
    try {
      const sloPipelinePromise = this.createPipeline((0, _sli_pipeline_template.getSLIPipelineTemplate)(updatedSlo, this.spaceId));
      rollbackOperations.push(() => this.deletePipeline((0, _constants.getSLOPipelineId)(updatedSlo.id, updatedSlo.revision)));
      const rollupTransformPromise = this.transformManager.install(updatedSlo);
      rollbackOperations.push(() => this.transformManager.uninstall(updatedRollupTransformId));
      const summaryPipelinePromise = this.createPipeline((0, _summary_pipeline_template.getSummaryPipelineTemplate)(updatedSlo, this.spaceId, this.basePath));
      rollbackOperations.push(() => this.deletePipeline((0, _constants.getSLOSummaryPipelineId)(updatedSlo.id, updatedSlo.revision)));
      const summaryTransformPromise = this.summaryTransformManager.install(updatedSlo);
      rollbackOperations.push(() => this.summaryTransformManager.uninstall(updatedSummaryTransformId));
      const tempDocPromise = this.createTempSummaryDocument(updatedSlo);
      rollbackOperations.push(() => this.deleteTempSummaryDocument(updatedSlo));
      await Promise.all([sloPipelinePromise, rollupTransformPromise, summaryPipelinePromise, summaryTransformPromise, tempDocPromise]);

      // transforms can only be started after the pipelines are created
      await Promise.all([this.transformManager.start(updatedRollupTransformId), this.summaryTransformManager.start(updatedSummaryTransformId)]);
    } catch (err) {
      var _err$meta2, _err$meta2$body, _err$meta2$body$error;
      this.logger.debug(`Cannot update the SLO [id: ${updatedSlo.id}, revision: ${updatedSlo.revision}]. Rolling back. ${err}`);
      await (0, _std.asyncForEach)(rollbackOperations.reverse(), async operation => {
        try {
          await operation();
        } catch (rollbackErr) {
          this.logger.debug(`Rollback operation failed. ${rollbackErr}`);
        }
      });
      if (((_err$meta2 = err.meta) === null || _err$meta2 === void 0 ? void 0 : (_err$meta2$body = _err$meta2.body) === null || _err$meta2$body === void 0 ? void 0 : (_err$meta2$body$error = _err$meta2$body.error) === null || _err$meta2$body$error === void 0 ? void 0 : _err$meta2$body$error.type) === 'security_exception') {
        throw new _errors.SecurityException(err.meta.body.error.reason);
      }
      throw err;
    }
    await this.deleteOriginalSLO(originalSlo);
    return this.toResponse(updatedSlo);
  }
  async isRevisionBumpRequired(originalSlo, updatedSlo) {
    const fields = ['indicator', 'groupBy', 'timeWindow', 'objective', 'budgetingMethod', 'settings'];
    const hasBreakingChanges = !(0, _lodash.isEqual)((0, _lodash.pick)(originalSlo, fields), (0, _lodash.pick)(updatedSlo, fields));
    const currentResourcesVersion = await this.summaryTransformManager.getVersion((0, _constants.getSLOSummaryTransformId)(originalSlo.id, originalSlo.revision));
    const hasOutdatedVersion = currentResourcesVersion === undefined || currentResourcesVersion < _constants.SLO_RESOURCES_VERSION;
    return hasBreakingChanges || hasOutdatedVersion;
  }
  async deleteOriginalSLO(slo) {
    try {
      await Promise.all([this.transformManager.uninstall((0, _constants.getSLOTransformId)(slo.id, slo.revision)), this.summaryTransformManager.uninstall((0, _constants.getSLOSummaryTransformId)(slo.id, slo.revision)), this.deletePipeline((0, _constants.getWildcardPipelineId)(slo.id, slo.revision))]);
      await Promise.all([this.deleteRollupData(slo), this.deleteSummaryData(slo)]);
    } catch (err) {
      // Any errors here should not prevent moving forward.
      // Worst case we keep rolling up data for the previous revision number.
    }
  }
  async deleteRollupData(slo) {
    await this.scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SLI_DESTINATION_INDEX_PATTERN,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': slo.id
            }
          }, {
            term: {
              'slo.revision': slo.revision
            }
          }]
        }
      }
    });
  }
  async deleteSummaryData(slo) {
    await this.scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      refresh: true,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': slo.id
            }
          }, {
            term: {
              'slo.revision': slo.revision
            }
          }]
        }
      }
    });
  }
  async deletePipeline(id) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.ingest.deletePipeline({
      id
    }, {
      ignore: [404]
    }), {
      logger: this.logger
    });
  }
  async createPipeline(params) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.ingest.putPipeline(params), {
      logger: this.logger
    });
  }
  async createTempSummaryDocument(slo) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asCurrentUser.index({
      index: _constants.SUMMARY_TEMP_INDEX_NAME,
      id: `slo-${slo.id}`,
      document: (0, _create_temp_summary.createTempSummaryDocument)(slo, this.spaceId, this.basePath),
      refresh: true
    }), {
      logger: this.logger
    });
  }
  async deleteTempSummaryDocument(slo) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asCurrentUser.delete({
      index: _constants.SUMMARY_TEMP_INDEX_NAME,
      id: `slo-${slo.id}`,
      refresh: true
    }), {
      logger: this.logger
    });
  }
  toResponse(slo) {
    return _sloSchema.updateSLOResponseSchema.encode(slo);
  }
}
exports.UpdateSLO = UpdateSLO;