"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runBulkDelete = runBulkDelete;
var _pLimit = _interopRequireDefault(require("p-limit"));
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function runBulkDelete(params, dependencies) {
  const {
    scopedClusterClient,
    rulesClient,
    deleteSLO,
    logger,
    abortController
  } = dependencies;
  logger.debug(`Starting bulk deletion for SLO [${params.list.join(', ')}]`);
  const limiter = (0, _pLimit.default)(3);
  const promises = params.list.map(async sloId => {
    return limiter(async () => {
      try {
        await deleteSLO.execute(sloId, {
          skipRuleDeletion: true,
          skipDataDeletion: true
        });
      } catch (err) {
        return {
          id: sloId,
          success: false,
          error: err.message
        };
      }
      return {
        id: sloId,
        success: true
      };
    });
  });
  const results = await Promise.all(promises);
  const itemsDeletedSuccessfully = results.filter(result => result.success === true).map(result => result.id);
  try {
    await Promise.all([scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SLI_DESTINATION_INDEX_PATTERN,
      refresh: false,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: {
            terms: {
              'slo.id': itemsDeletedSuccessfully
            }
          }
        }
      }
    }, {
      signal: abortController.signal
    }), scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      refresh: false,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: {
            terms: {
              'slo.id': itemsDeletedSuccessfully
            }
          }
        }
      }
    }, {
      signal: abortController.signal
    })]);
  } catch (err) {
    logger.debug(`Error scheduling tasks for data deletion: ${err}`);
  }
  try {
    await rulesClient.bulkDeleteRules({
      filter: `alert.attributes.params.sloId:(${itemsDeletedSuccessfully.join(' or ')})`
    });
  } catch (err) {
    logger.debug(`Error deleting rules: ${err}`);
  }
  logger.debug(`completed bulk deletion: [${itemsDeletedSuccessfully.join(',')}]`);
  return results;
}