"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchProcessWithIOEvents = exports.registerIOEventsRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerIOEventsRoute = (router, logger) => {
  router.versioned.get({
    access: 'internal',
    path: _constants.IO_EVENTS_ROUTE
  }).addVersion({
    version: '1',
    security: {
      authz: {
        enabled: false,
        reason: `This route delegates authorization to Elasticsearch and it's not tied to a Kibana privilege.`
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          index: _configSchema.schema.string(),
          sessionEntityId: _configSchema.schema.string(),
          sessionStartTime: _configSchema.schema.string(),
          cursor: _configSchema.schema.maybe(_configSchema.schema.string()),
          pageSize: _configSchema.schema.maybe(_configSchema.schema.number({
            min: 1,
            max: _constants.IO_EVENTS_PER_PAGE
          })) // currently only set in FTR tests to test pagination
        })
      }
    }
  }, async (context, request, response) => {
    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const {
      index,
      sessionEntityId,
      sessionStartTime,
      cursor,
      pageSize = _constants.IO_EVENTS_PER_PAGE
    } = request.query;
    try {
      var _search$hits$total;
      const search = await client.search({
        index: [index],
        body: {
          query: {
            bool: {
              must: [{
                term: {
                  [_constants.ENTRY_SESSION_ENTITY_ID_PROPERTY]: sessionEntityId
                }
              }, {
                term: {
                  [_ruleDataUtils.EVENT_ACTION]: 'text_output'
                }
              }, {
                range: {
                  // optimization to prevent data before this session from being hit.
                  [_constants.TIMESTAMP_PROPERTY]: {
                    gte: sessionStartTime
                  }
                }
              }]
            }
          },
          size: Math.min(pageSize, _constants.IO_EVENTS_PER_PAGE),
          sort: [{
            [_ruleDataUtils.TIMESTAMP]: 'asc'
          }],
          search_after: cursor ? [cursor] : undefined,
          fields: _constants.IO_EVENT_FIELDS
        }
      });
      const events = search.hits.hits;
      const total = typeof search.hits.total === 'number' ? search.hits.total : (_search$hits$total = search.hits.total) === null || _search$hits$total === void 0 ? void 0 : _search$hits$total.value;
      return response.ok({
        body: {
          total,
          events
        }
      });
    } catch (err) {
      var _err$meta;
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      logger.error(`Failed to fetch io events: ${err}`);

      // unauthorized
      if ((err === null || err === void 0 ? void 0 : (_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : _err$meta.statusCode) === 403) {
        return response.ok({
          body: {
            total: 0,
            events: []
          }
        });
      }
      return response.customError({
        body: {
          message: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.registerIOEventsRoute = registerIOEventsRoute;
const searchProcessWithIOEvents = async (client, index, sessionEntityId, range) => {
  const rangeFilter = range ? [{
    range: {
      [_ruleDataUtils.TIMESTAMP]: {
        gte: range[0],
        lte: range[1]
      }
    }
  }] : [];
  try {
    var _search$aggregations;
    const search = await client.search({
      index: [index],
      body: {
        query: {
          bool: {
            must: [{
              term: {
                [_ruleDataUtils.EVENT_ACTION]: 'text_output'
              }
            }, {
              term: {
                [_constants.ENTRY_SESSION_ENTITY_ID_PROPERTY]: sessionEntityId
              }
            }, ...rangeFilter]
          }
        },
        size: 0,
        aggs: {
          custom_agg: {
            terms: {
              field: _constants.PROCESS_ENTITY_ID_PROPERTY,
              size: _constants.PROCESS_EVENTS_PER_PAGE
            }
          }
        }
      }
    });
    const agg = (_search$aggregations = search.aggregations) === null || _search$aggregations === void 0 ? void 0 : _search$aggregations.custom_agg;
    const buckets = (agg === null || agg === void 0 ? void 0 : agg.buckets) || [];
    return buckets.map(bucket => ({
      _source: {
        event: {
          kind: 'event',
          action: 'text_output',
          id: bucket.key
        },
        process: {
          entity_id: bucket.key
        }
      }
    }));
  } catch (err) {
    return [];
  }
};
exports.searchProcessWithIOEvents = searchProcessWithIOEvents;