"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUsageRecords = exports.getSearchQueryByCloudSecuritySolution = exports.getCloudSecurityUsageRecord = exports.getAssetAggQueryByCloudSecuritySolution = exports.getAssetAggByCloudSecuritySolution = exports.getAggregationByCloudSecuritySolution = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getUsageRecords = (assetCountAggregation, cloudSecuritySolution, taskId, tier, projectId, periodSeconds, logger) => {
  let assetCount;
  let resourceSubtypeCounterMap;
  if (cloudSecuritySolution === _constants.CSPM || cloudSecuritySolution === _constants.KSPM) {
    const resourceSubtypeBuckets = assetCountAggregation.resource_sub_type.buckets;
    const billableAssets = _constants.BILLABLE_ASSETS_CONFIG[cloudSecuritySolution].values;
    assetCount = resourceSubtypeBuckets.filter(bucket => billableAssets.includes(bucket.key)).reduce((acc, bucket) => acc + bucket.unique_assets.value, 0);
    resourceSubtypeCounterMap = assetCountAggregation.resource_sub_type.buckets.reduce((resourceMap, item) => {
      // By the usage spec, the resource subtype counter should be a string // https://github.com/elastic/usage-api/blob/main/api/user-v1-spec.yml
      resourceMap[item.key] = String(item.unique_assets.value);
      return resourceMap;
    }, {});
  } else {
    assetCount = assetCountAggregation.unique_assets.value;
  }
  if (assetCount > _constants.AGGREGATION_PRECISION_THRESHOLD) {
    logger.warn(`The number of unique resources for {${cloudSecuritySolution}} is ${assetCount}, which is higher than the AGGREGATION_PRECISION_THRESHOLD of ${_constants.AGGREGATION_PRECISION_THRESHOLD}.`);
  }
  const minTimestamp = new Date(assetCountAggregation.min_timestamp.value_as_string).toISOString();
  const creationTimestamp = new Date();
  const minutes = creationTimestamp.getMinutes();
  if (minutes >= 30) {
    creationTimestamp.setMinutes(30, 0, 0);
  } else {
    creationTimestamp.setMinutes(0, 0, 0);
  }
  const roundedCreationTimestamp = creationTimestamp.toISOString();
  const usageRecord = {
    id: `${_constants.CLOUD_SECURITY_TASK_TYPE}_${cloudSecuritySolution}_${projectId}_${roundedCreationTimestamp}`,
    usage_timestamp: minTimestamp,
    creation_timestamp: creationTimestamp.toISOString(),
    usage: {
      type: _constants.CLOUD_SECURITY_TASK_TYPE,
      sub_type: cloudSecuritySolution,
      quantity: assetCount,
      period_seconds: periodSeconds,
      ...(resourceSubtypeCounterMap && {
        metadata: resourceSubtypeCounterMap
      })
    },
    source: {
      id: taskId,
      instance_group_id: projectId,
      metadata: {
        tier
      }
    }
  };
  return usageRecord;
};
exports.getUsageRecords = getUsageRecords;
const getAggregationByCloudSecuritySolution = cloudSecuritySolution => {
  if (cloudSecuritySolution === _constants.CSPM || cloudSecuritySolution === _constants.KSPM) return {
    resource_sub_type: {
      terms: {
        field: _constants.BILLABLE_ASSETS_CONFIG[cloudSecuritySolution].filter_attribute
      },
      aggs: {
        unique_assets: {
          cardinality: {
            field: _constants.METERING_CONFIGS[cloudSecuritySolution].assets_identifier,
            precision_threshold: _constants.AGGREGATION_PRECISION_THRESHOLD
          }
        }
      }
    },
    min_timestamp: {
      min: {
        field: '@timestamp'
      }
    }
  };
  return {
    unique_assets: {
      cardinality: {
        field: _constants.METERING_CONFIGS[cloudSecuritySolution].assets_identifier,
        precision_threshold: _constants.AGGREGATION_PRECISION_THRESHOLD
      }
    },
    min_timestamp: {
      min: {
        field: '@timestamp'
      }
    }
  };
};
exports.getAggregationByCloudSecuritySolution = getAggregationByCloudSecuritySolution;
const getSearchQueryByCloudSecuritySolution = cloudSecuritySolution => {
  const mustFilters = [];
  if (cloudSecuritySolution === _constants.CSPM || cloudSecuritySolution === _constants.KSPM || cloudSecuritySolution === _constants.CNVM) {
    mustFilters.push({
      range: {
        '@timestamp': {
          gte: `now-${_constants.ASSETS_SAMPLE_GRANULARITY}`
        }
      }
    });
  }
  if (cloudSecuritySolution === _constants.CSPM || cloudSecuritySolution === _constants.KSPM) {
    mustFilters.push({
      term: {
        'rule.benchmark.posture_type': cloudSecuritySolution
      }
    });
  }
  return {
    bool: {
      must: mustFilters
    }
  };
};
exports.getSearchQueryByCloudSecuritySolution = getSearchQueryByCloudSecuritySolution;
const getAssetAggQueryByCloudSecuritySolution = cloudSecuritySolution => {
  const query = getSearchQueryByCloudSecuritySolution(cloudSecuritySolution);
  const aggs = getAggregationByCloudSecuritySolution(cloudSecuritySolution);
  return {
    index: _constants.METERING_CONFIGS[cloudSecuritySolution].index,
    query,
    size: 0,
    aggs
  };
};
exports.getAssetAggQueryByCloudSecuritySolution = getAssetAggQueryByCloudSecuritySolution;
const getAssetAggByCloudSecuritySolution = async (esClient, cloudSecuritySolution) => {
  const assetsAggQuery = getAssetAggQueryByCloudSecuritySolution(cloudSecuritySolution);
  const response = await esClient.search(assetsAggQuery);
  if (!response.aggregations) return;
  return response.aggregations;
};
exports.getAssetAggByCloudSecuritySolution = getAssetAggByCloudSecuritySolution;
const indexHasDataInDateRange = async (esClient, cloudSecuritySolution) => {
  var _response$hits;
  const response = await esClient.search({
    index: _constants.METERING_CONFIGS[cloudSecuritySolution].index,
    size: 1,
    _source: false,
    query: getSearchQueryByCloudSecuritySolution(cloudSecuritySolution)
  }, {
    ignore: [404]
  });
  return ((_response$hits = response.hits) === null || _response$hits === void 0 ? void 0 : _response$hits.hits.length) > 0;
};
const getCloudSecurityUsageRecord = async ({
  esClient,
  projectId,
  taskId,
  cloudSecuritySolution,
  tier,
  logger
}) => {
  try {
    if (!(await indexHasDataInDateRange(esClient, cloudSecuritySolution))) return;

    // const periodSeconds = Math.floor((new Date().getTime() - searchFrom.getTime()) / 1000);
    const periodSeconds = 1800; // Workaround to prevent overbilling by charging for a constant time window. The issue should be resolved in https://github.com/elastic/security-team/issues/9424.

    const assetCountAggregation = await getAssetAggByCloudSecuritySolution(esClient, cloudSecuritySolution);
    if (!assetCountAggregation) return [];
    const usageRecords = await getUsageRecords(assetCountAggregation, cloudSecuritySolution, taskId, tier, projectId, periodSeconds, logger);
    return [usageRecords];
  } catch (err) {
    logger.error(`Failed to fetch ${cloudSecuritySolution} metering data ${err}`);
  }
};
exports.getCloudSecurityUsageRecord = getCloudSecurityUsageRecord;