"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CircuitBreakingQueryExecutorImpl = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = _interopRequireWildcard(require("rxjs"));
var rx = _rxjs;
var _health_diagnostic_circuit_breakers = require("./health_diagnostic_circuit_breakers.types");
var _health_diagnostic_service = require("./health_diagnostic_service.types");
var _helpers = require("../helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CircuitBreakingQueryExecutorImpl {
  constructor(client, logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.client = client;
    this.logger = (0, _helpers.newTelemetryLogger)(logger.get('circuit-breaking-query-executor'));
  }
  search({
    query,
    circuitBreakers
  }) {
    const controller = new AbortController();
    const abortSignal = controller.signal;
    const circuitBreakers$ = this.configureCircuitBreakers(circuitBreakers, controller);
    switch (query.type) {
      case _health_diagnostic_service.QueryType.DSL:
        return this.streamDSL(query, abortSignal).pipe((0, _rxjs.takeUntil)(circuitBreakers$));
      case _health_diagnostic_service.QueryType.EQL:
        return this.streamEql(query, abortSignal).pipe((0, _rxjs.takeUntil)(circuitBreakers$));
      case _health_diagnostic_service.QueryType.ESQL:
        return this.streamEsql(query, abortSignal).pipe((0, _rxjs.takeUntil)(circuitBreakers$));
      default:
        {
          const exhaustiveCheck = query.type;
          throw new Error(`Unhandled QueryType: ${exhaustiveCheck}`);
        }
    }
  }
  streamEsql(diagnosticQuery, abortSignal) {
    const regex = /^[\s\r\n]*FROM/;
    return (0, _rxjs.from)(this.indicesFor(diagnosticQuery)).pipe((0, _rxjs.mergeMap)(index => {
      const query = regex.test(diagnosticQuery.query) ? diagnosticQuery.query : `FROM ${index} | ${diagnosticQuery.query}`;
      return (0, _rxjs.from)(this.client.helpers.esql({
        query
      }, {
        signal: abortSignal
      }).toRecords()).pipe((0, _rxjs.mergeMap)(resp => {
        return resp.records.map(r => r);
      }));
    }));
  }
  streamEql(diagnosticQuery, abortSignal) {
    return (0, _rxjs.from)(this.indicesFor(diagnosticQuery)).pipe((0, _rxjs.mergeMap)(index => {
      const request = {
        index,
        query: diagnosticQuery.query,
        size: diagnosticQuery.size
      };
      return (0, _rxjs.from)(this.client.eql.search(request, {
        signal: abortSignal
      })).pipe((0, _rxjs.mergeMap)(resp => {
        if (resp.hits.events) {
          return resp.hits.events.map(h => h._source);
        } else if (resp.hits.sequences) {
          return resp.hits.sequences.map(seq => seq.events.map(h => h._source));
        } else {
          this.logger.warn('>> Neither hits.events nor hits.sequences found in the response for query', {
            queryName: diagnosticQuery.name
          });
          return [];
        }
      }));
    }));
  }
  streamDSL(diagnosticQuery, abortSignal, pitKeepAlive = '1m') {
    var _diagnosticQuery$size;
    let pitId;
    let searchAfter;
    const pageSize = (_diagnosticQuery$size = diagnosticQuery.size) !== null && _diagnosticQuery$size !== void 0 ? _diagnosticQuery$size : 10000;
    const query = JSON.parse(diagnosticQuery.query);
    const fetchPage = () => {
      const paginatedRequest = {
        size: pageSize,
        sort: [{
          _shard_doc: 'asc'
        }],
        search_after: searchAfter,
        pit: {
          id: pitId,
          keep_alive: pitKeepAlive
        },
        ...query
      };
      return this.client.search(paginatedRequest, {
        signal: abortSignal
      });
    };
    return (0, _rxjs.from)(this.indicesFor(diagnosticQuery)).pipe((0, _rxjs.mergeMap)(index => (0, _rxjs.from)(this.client.openPointInTime({
      index,
      keep_alive: pitKeepAlive
    }))), (0, _rxjs.map)(res => res.id), (0, _rxjs.mergeMap)(id => {
      pitId = id;
      return (0, _rxjs.from)(fetchPage());
    }), (0, _rxjs.expand)(searchResponse => {
      const hits = searchResponse.hits.hits;
      const aggrs = searchResponse.aggregations;
      if (aggrs || hits.length === 0) {
        return _rxjs.EMPTY;
      }
      searchAfter = hits[hits.length - 1].sort;
      return (0, _rxjs.from)(fetchPage());
    }), (0, _rxjs.mergeMap)(searchResponse => {
      if (searchResponse.aggregations) {
        return [searchResponse.aggregations];
      } else {
        return searchResponse.hits.hits.map(h => h._source);
      }
    }), (0, _rxjs.finalize)(() => {
      this.client.closePointInTime({
        id: pitId
      }).catch(error => {
        this.logger.warn('>> closePointInTime error', {
          error
        });
      });
    }));
  }
  configureCircuitBreakers(circuitBreakers, controller) {
    return (0, _rxjs.merge)(...circuitBreakers.map(cb => (0, _rxjs.timer)(0, cb.validationIntervalMs()).pipe(rx.mergeMap(() => rx.from(cb.validate())), (0, _rxjs.filter)(result => !result.valid)))).pipe((0, _rxjs.map)(result => {
      this.logger.debug('>> Circuit breaker triggered', {
        circuitBreaker: result
      });
      controller.abort();
      throw new _health_diagnostic_circuit_breakers.ValidationError(result);
    }));
  }

  /**
   * Returns the list of indices to query based on the provided tiers.
   * When running in serverless or `query.index` is not managed by an ILM, returns
   * the same `query.index`.
   *
   * @param query The health diagnostic query object.
   * @returns A Promise resolving to an array of indices.
   */
  async indicesFor(query) {
    if (query.tiers === undefined) {
      this.logger.debug('No tiers defined in the query, returning index as is', {
        queryName: query.name
      });
      return [query.index];
    }
    const tiers = query.tiers;
    return (await this.client.ilm.explainLifecycle({
      index: query.index,
      only_managed: false,
      filter_path: ['indices.*.phase']
    }).then(response => {
      if (response.indices === undefined) {
        this.logger.debug('Got an empty response while explaining lifecycle. Asumming serverless.', {
          index: query.index
        });
        return [query.index];
      } else {
        const indices = Object.entries(response.indices).map(([indexName, stats]) => {
          if ('phase' in stats && stats.phase) {
            if (tiers.includes(stats.phase)) {
              return indexName;
            } else {
              this.logger.debug('Index is not in the expected phases', {
                phase: stats.phase,
                index: indexName,
                tiers
              });
              return '';
            }
          } else {
            // should not happen, but just in case
            this.logger.debug('Index is not managed by an ILM', {
              index: indexName,
              tiers
            });
            return '';
          }
        });
        this.logger.debug('Indices managed by ILM', {
          queryName: query.name,
          tiers: query.tiers,
          indices
        });
        return indices;
      }
    })).filter(indexName => indexName !== '');
  }
}
exports.CircuitBreakingQueryExecutorImpl = CircuitBreakingQueryExecutorImpl;