"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsResourceUpsertRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _partition = _interopRequireDefault(require("lodash/partition"));
var _constants = require("../../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _resources = require("../../../../../../common/siem_migrations/rules/resources");
var _audit = require("../util/audit");
var _authz = require("../util/authz");
var _lookups = require("../util/lookups");
var _with_license = require("../util/with_license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsResourceUpsertRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_RULE_MIGRATION_RESOURCES_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    },
    options: {
      body: {
        maxBytes: 26214400
      }
    } // rise payload limit to 25MB
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.UpsertRuleMigrationResourcesRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.UpsertRuleMigrationResourcesRequestBody)
      }
    }
  }, (0, _with_license.withLicense)(async (context, req, res) => {
    const resources = req.body;
    const migrationId = req.params.migration_id;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution);
    try {
      const ctx = await context.resolve(['securitySolution']);
      const ruleMigrationsClient = ctx.securitySolution.getSiemRuleMigrationsClient();
      await siemMigrationAuditLogger.logUploadResources({
        migrationId
      });

      // Check if the migration exists
      const {
        data
      } = await ruleMigrationsClient.data.rules.get(migrationId, {
        size: 1
      });
      const [rule] = data;
      if (!rule) {
        return res.notFound({
          body: {
            message: 'Migration not found'
          }
        });
      }
      const [lookups, macros] = (0, _partition.default)(resources, {
        type: 'lookup'
      });
      const processedLookups = await (0, _lookups.processLookups)(lookups, ruleMigrationsClient);
      const resourcesUpsert = [...macros, ...processedLookups].map(resource => ({
        ...resource,
        migration_id: migrationId
      }));

      // Upsert the resources
      await ruleMigrationsClient.data.resources.upsert(resourcesUpsert);

      // Create identified resource documents to keep track of them (without content)
      const resourceIdentifier = new _resources.ResourceIdentifier(rule.original_rule.vendor);
      const resourcesToCreate = resourceIdentifier.fromResources(resources).map(resource => ({
        ...resource,
        migration_id: migrationId
      }));
      await ruleMigrationsClient.data.resources.create(resourcesToCreate);
      return res.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (error) {
      logger.error(error);
      await siemMigrationAuditLogger.logUploadResources({
        migrationId,
        error
      });
      return res.badRequest({
        body: error.message
      });
    }
  }));
};
exports.registerSiemRuleMigrationsResourceUpsertRoute = registerSiemRuleMigrationsResourceUpsertRoute;