"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDetectionEngineHealthClient = void 0;
var _with_security_span = require("../../../../../utils/with_security_span");
var _install_assets_for_rule_monitoring = require("./assets/install_assets_for_rule_monitoring");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createDetectionEngineHealthClient = (ruleSpacesClient, ruleObjectsHealthClient, eventLogHealthClient, savedObjectsImporter, logger) => {
  const currentSpaceId = ruleSpacesClient.getCurrentSpaceId();
  return {
    calculateRuleHealth: args => {
      return (0, _with_security_span.withSecuritySpan)('IDetectionEngineHealthClient.calculateRuleHealth', async () => {
        const ruleId = args.rule_id;
        try {
          // We call these two sequentially, because if the rule doesn't exist we need to throw 404
          // from ruleObjectsHealthClient before we calculate expensive stats in eventLogHealthClient.
          const statsBasedOnRuleObjects = await ruleObjectsHealthClient.calculateRuleHealth(args);
          const statsBasedOnEventLog = await eventLogHealthClient.calculateRuleHealth(args);
          return {
            state_at_the_moment: statsBasedOnRuleObjects.state_at_the_moment,
            stats_over_interval: statsBasedOnEventLog.stats_over_interval,
            history_over_interval: statsBasedOnEventLog.history_over_interval,
            debug: {
              ...statsBasedOnRuleObjects.debug,
              ...statsBasedOnEventLog.debug
            }
          };
        } catch (e) {
          const logMessage = 'Error calculating rule health';
          const logReason = e instanceof Error ? e.message : String(e);
          const logSuffix = `[rule id ${ruleId}][space id ${currentSpaceId}]`;
          const logMeta = {
            rule: {
              id: ruleId
            },
            kibana: {
              spaceId: currentSpaceId
            }
          };
          logger.error(`${logMessage}: ${logReason} ${logSuffix}`, logMeta);
          throw e;
        }
      });
    },
    calculateSpaceHealth: args => {
      return (0, _with_security_span.withSecuritySpan)('IDetectionEngineHealthClient.calculateSpaceHealth', async () => {
        try {
          const [statsBasedOnRuleObjects, statsBasedOnEventLog] = await Promise.all([ruleObjectsHealthClient.calculateSpaceHealth(args), eventLogHealthClient.calculateSpaceHealth(args)]);
          return {
            state_at_the_moment: statsBasedOnRuleObjects.state_at_the_moment,
            stats_over_interval: statsBasedOnEventLog.stats_over_interval,
            history_over_interval: statsBasedOnEventLog.history_over_interval,
            debug: {
              ...statsBasedOnRuleObjects.debug,
              ...statsBasedOnEventLog.debug
            }
          };
        } catch (e) {
          const logMessage = 'Error calculating space health';
          const logReason = e instanceof Error ? e.message : String(e);
          const logSuffix = `[space id ${currentSpaceId}]`;
          const logMeta = {
            kibana: {
              spaceId: currentSpaceId
            }
          };
          logger.error(`${logMessage}: ${logReason} ${logSuffix}`, logMeta);
          throw e;
        }
      });
    },
    calculateClusterHealth: args => {
      return (0, _with_security_span.withSecuritySpan)('IDetectionEngineHealthClient.calculateClusterHealth', async () => {
        try {
          const [statsBasedOnRuleObjects, statsBasedOnEventLog] = await Promise.all([ruleObjectsHealthClient.calculateClusterHealth(args), eventLogHealthClient.calculateClusterHealth(args)]);
          return {
            state_at_the_moment: statsBasedOnRuleObjects.state_at_the_moment,
            stats_over_interval: statsBasedOnEventLog.stats_over_interval,
            history_over_interval: statsBasedOnEventLog.history_over_interval,
            debug: {
              ...statsBasedOnRuleObjects.debug,
              ...statsBasedOnEventLog.debug
            }
          };
        } catch (e) {
          const logMessage = 'Error calculating cluster health';
          const logReason = e instanceof Error ? e.message : String(e);
          const logSuffix = `[space id ${currentSpaceId}]`;
          const logMeta = {
            kibana: {
              spaceId: currentSpaceId
            }
          };
          logger.error(`${logMessage}: ${logReason} ${logSuffix}`, logMeta);
          throw e;
        }
      });
    },
    installAssetsForMonitoringHealth: () => {
      return (0, _with_security_span.withSecuritySpan)('IDetectionEngineHealthClient.installAssetsForMonitoringHealth', async () => {
        try {
          await (0, _install_assets_for_rule_monitoring.installAssetsForRuleMonitoring)(savedObjectsImporter, logger, currentSpaceId);
        } catch (e) {
          const logMessage = 'Error installing assets for monitoring Detection Engine health';
          const logReason = e instanceof Error ? e.message : String(e);
          const logSuffix = `[space id ${currentSpaceId}]`;
          const logMeta = {
            kibana: {
              spaceId: currentSpaceId
            }
          };
          logger.error(`${logMessage}: ${logReason} ${logSuffix}`, logMeta);
          throw e;
        }
      });
    }
  };
};
exports.createDetectionEngineHealthClient = createDetectionEngineHealthClient;