"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateRuleActions = void 0;
var _lodash = require("lodash");
var _boom = require("@hapi/boom");
var _utils = require("../../../../routes/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getActionsOrError = async ({
  actionsClient
}) => {
  try {
    return {
      allActions: await actionsClient.getAll({
        includeSystemActions: true
      }),
      bulkError: undefined
    };
  } catch (exc) {
    if ((0, _boom.isBoom)(exc) && exc.output.statusCode === 403) {
      return {
        allActions: undefined,
        bulkError: (0, _utils.createBulkErrorObject)({
          statusCode: 403,
          message: `You may not have actions privileges required to import rules with actions: ${exc.output.payload.message}`
        })
      };
    } else {
      throw exc;
    }
  }
};
const validateRuleActions = async ({
  actionsClient,
  rules
}) => {
  const [rulesWithActions, rulesWithoutActions] = (0, _lodash.partition)(rules, rule => rule.actions != null && rule.actions.length > 0);
  if (rulesWithActions.length === 0) {
    return {
      validatedActionRules: rulesWithoutActions,
      missingActionErrors: []
    };
  }
  const missingActionErrors = [];
  const actionsOrError = await getActionsOrError({
    actionsClient
  });
  if (actionsOrError.bulkError != null) {
    return {
      validatedActionRules: rulesWithoutActions,
      missingActionErrors: rulesWithActions.map(rule => ({
        id: rule.id,
        rule_id: rule.rule_id,
        error: actionsOrError.bulkError.error
      }))
    };
  }
  const allActionIdsSet = new Set(actionsOrError.allActions.map(action => action.id));
  const validatedRulesWithActions = rulesWithActions.filter(rule => {
    // We know rulesWithActions have actions, but TypeScript does not
    if (rule.actions == null || rule.actions.length === 0) {
      return true;
    }
    const missingActions = rule.actions.filter(action => !allActionIdsSet.has(action.id));
    if (missingActions.length > 0) {
      missingActionErrors.push({
        id: rule.id,
        rule_id: rule.rule_id,
        error: {
          status_code: 404,
          message: `Rule actions reference the following missing action IDs: ${missingActions.map(action => action.id).join(',')}`
        }
      });
      return false;
    }
    return true;
  });
  return {
    validatedActionRules: rulesWithoutActions.concat(validatedRulesWithActions),
    missingActionErrors
  };
};
exports.validateRuleActions = validateRuleActions;