"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkScheduleRuleGapFilling = void 0;
var _lodash = require("lodash");
var _validations = require("../../../logic/bulk_actions/validations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkScheduleRuleGapFilling = async ({
  rules,
  isDryRun,
  rulesClient,
  mlAuthz,
  fillGapsPayload
}) => {
  const errors = [];
  // In the first step, we validate if it is possible to schedule backfill for the rules
  const validationResults = await Promise.all(rules.map(async rule => {
    try {
      await (0, _validations.validateBulkRuleGapFilling)({
        mlAuthz,
        rule
      });
      return {
        valid: true,
        rule
      };
    } catch (error) {
      return {
        valid: false,
        rule,
        error
      };
    }
  }));
  const validatedRules = validationResults.filter(({
    valid
  }) => valid).map(({
    rule
  }) => rule);
  errors.push(...validationResults.filter(({
    valid
  }) => !valid).map(({
    rule,
    error
  }) => ({
    item: rule,
    error
  })));
  if (isDryRun || validatedRules.length === 0) {
    return {
      backfilled: validatedRules,
      errors,
      skipped: []
    };
  }
  const {
    start_date: start,
    end_date: end
  } = fillGapsPayload;

  // Due to performance considerations we will backfill a maximum of 1000 gaps per rule when called with many rules
  // however, this endpoint will be called with one rule as well. In that case, we will increase the limit to 10_000
  // in order to attempt to fill all the gaps of the rule in the specified time range
  const maxGapCountPerRule = rules.length === 1 ? 10_000 : 1000;
  const {
    backfilled,
    skipped,
    errored
  } = await rulesClient.bulkFillGapsByRuleIds({
    rules: validatedRules.map(({
      id,
      name,
      consumer,
      alertTypeId
    }) => ({
      id,
      name,
      consumer,
      alertTypeId
    })),
    range: {
      start,
      end
    }
  }, {
    maxGapCountPerRule
  });
  errored.forEach(backfillingError => {
    errors.push({
      rule: backfillingError.rule,
      message: `${backfillingError.step} - ${backfillingError.errorMessage}`
    });
  });
  const rulesDict = (0, _lodash.keyBy)(validatedRules, 'id');
  return {
    backfilled: backfilled.map(({
      id
    }) => rulesDict[id]).filter(_lodash.identity),
    errors,
    skipped
  };
};
exports.bulkScheduleRuleGapFilling = bulkScheduleRuleGapFilling;