"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setSignalsStatusRoute = void 0;
var _lodash = require("lodash");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _signals = require("../../../../../common/api/detection_engine/signals");
var _constants = require("../../../../../common/constants");
var _utils = require("../utils");
var _constants2 = require("../../../telemetry/constants");
var _insights = require("../../../telemetry/insights");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const setSignalsStatusRoute = (router, logger, sender) => {
  router.versioned.post({
    path: _constants.DETECTION_ENGINE_SIGNALS_STATUS_URL,
    access: 'public',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_signals.SetAlertsStatusRequestBody)
      }
    }
  }, async (context, request, response) => {
    var _securitySolution$get;
    const {
      status
    } = request.body;
    const core = await context.core;
    const securitySolution = await context.securitySolution;
    const esClient = core.elasticsearch.client.asCurrentUser;
    const siemClient = securitySolution === null || securitySolution === void 0 ? void 0 : securitySolution.getAppClient();
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const spaceId = (_securitySolution$get = securitySolution === null || securitySolution === void 0 ? void 0 : securitySolution.getSpaceId()) !== null && _securitySolution$get !== void 0 ? _securitySolution$get : 'default';
    if (!siemClient) {
      return siemResponse.error({
        statusCode: 404
      });
    }
    const user = core.security.authc.getCurrentUser();
    const clusterId = sender.getClusterID();
    const isTelemetryOptedIn = await sender.isTelemetryOptedIn();
    if (isTelemetryOptedIn && clusterId) {
      // Sometimes the ids are in the query not passed in the request?
      const toSendAlertIds = 'signal_ids' in request.body ? request.body.signal_ids : (0, _lodash.get)(request.body.query, 'bool.filter.terms._id');
      // Get Context for Insights Payloads
      const sessionId = (0, _insights.getSessionIDfromKibanaRequest)(clusterId, request);
      if (user !== null && user !== void 0 && user.username && toSendAlertIds && sessionId && status) {
        const insightsPayloads = (0, _insights.createAlertStatusPayloads)(clusterId, toSendAlertIds, sessionId, user.username, _constants.DETECTION_ENGINE_SIGNALS_STATUS_URL, status);
        logger.debug(() => `Sending Insights Payloads ${JSON.stringify(insightsPayloads)}`);
        await sender.sendOnDemand(_constants2.INSIGHTS_CHANNEL, insightsPayloads);
      }
    }
    try {
      if ('signal_ids' in request.body) {
        const {
          signal_ids: signalIds
        } = request.body;
        const body = await updateSignalsStatusByIds(status, signalIds, spaceId, esClient, user);
        return response.ok({
          body
        });
      } else {
        const {
          conflicts,
          query
        } = request.body;
        const body = await updateSignalsStatusByQuery(status, query, {
          conflicts: conflicts !== null && conflicts !== void 0 ? conflicts : 'abort'
        }, spaceId, esClient, user);
        return response.ok({
          body
        });
      }
    } catch (err) {
      // error while getting or updating signal with id: id in signal index .siem-signals
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.setSignalsStatusRoute = setSignalsStatusRoute;
const updateSignalsStatusByIds = async (status, signalsId, spaceId, esClient, user) => esClient.updateByQuery({
  index: `${_constants.DEFAULT_ALERTS_INDEX}-${spaceId}`,
  refresh: true,
  body: {
    script: getUpdateSignalStatusScript(status, user),
    query: {
      bool: {
        filter: {
          terms: {
            _id: signalsId
          }
        }
      }
    }
  },
  ignore_unavailable: true
});

/**
 * Please avoid using `updateSignalsStatusByQuery` when possible, use `updateSignalsStatusByIds` instead.
 *
 * This method calls `updateByQuery` with `refresh: true` which is expensive on serverless.
 */
const updateSignalsStatusByQuery = async (status, query, options, spaceId, esClient, user) => esClient.updateByQuery({
  index: `${_constants.DEFAULT_ALERTS_INDEX}-${spaceId}`,
  conflicts: options.conflicts,
  refresh: true,
  body: {
    script: getUpdateSignalStatusScript(status, user),
    query: {
      bool: {
        filter: query
      }
    }
  },
  ignore_unavailable: true
});
const getUpdateSignalStatusScript = (status, user) => ({
  source: `if (ctx._source['${_ruleDataUtils.ALERT_WORKFLOW_STATUS}'] != null && ctx._source['${_ruleDataUtils.ALERT_WORKFLOW_STATUS}'] != '${status}') {
      ctx._source['${_ruleDataUtils.ALERT_WORKFLOW_STATUS}'] = '${status}';
      ctx._source['${_ruleDataUtils.ALERT_WORKFLOW_USER}'] = ${user !== null && user !== void 0 && user.profile_uid ? `'${user.profile_uid}'` : 'null'};
      ctx._source['${_ruleDataUtils.ALERT_WORKFLOW_STATUS_UPDATED_AT}'] = '${new Date().toISOString()}';
    }
    if (ctx._source.signal != null && ctx._source.signal.status != null) {
      ctx._source.signal.status = '${status}'
    }`,
  lang: 'painless'
});