"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleTypeDiffAlgorithm = void 0;
var _utility_types = require("../../../../../../../../common/utility_types");
var _prebuilt_rules = require("../../../../../../../../common/api/detection_engine/prebuilt_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ruleTypeDiffAlgorithm = versions => {
  const {
    base_version: baseVersion,
    current_version: currentVersion,
    target_version: targetVersion
  } = versions;
  const diffOutcome = (0, _prebuilt_rules.determineDiffOutcome)(baseVersion, currentVersion, targetVersion);
  const valueCanUpdate = (0, _prebuilt_rules.determineIfValueCanUpdate)(diffOutcome);
  const hasBaseVersion = baseVersion !== _prebuilt_rules.MissingVersion;
  const {
    mergeOutcome,
    conflict,
    mergedVersion
  } = mergeVersions({
    targetVersion,
    diffOutcome
  });
  return {
    has_base_version: hasBaseVersion,
    base_version: hasBaseVersion ? baseVersion : undefined,
    current_version: currentVersion,
    target_version: targetVersion,
    merged_version: mergedVersion,
    merge_outcome: mergeOutcome,
    diff_outcome: diffOutcome,
    has_update: valueCanUpdate,
    conflict
  };
};
exports.ruleTypeDiffAlgorithm = ruleTypeDiffAlgorithm;
const mergeVersions = ({
  targetVersion,
  diffOutcome
}) => {
  switch (diffOutcome) {
    // Missing base versions always return target version
    // Scenario -AA is treated as AAA
    // https://github.com/elastic/kibana/issues/210358#issuecomment-2654492854
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueNoUpdate:
      return {
        conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
        mergedVersion: targetVersion,
        mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
      };
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueSameUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueCanUpdate:
    // NOTE: This scenario is currently inaccessible via normal UI or API workflows, but the logic is covered just in case
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueCanUpdate:
    // Missing base versions always return target version
    // We return all -AB rule type fields as NON_SOLVABLE, whether or not the rule is customized
    // https://github.com/elastic/kibana/issues/210358#issuecomment-2654492854
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseCanUpdate:
      {
        return {
          mergedVersion: targetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target,
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NON_SOLVABLE
        };
      }
    default:
      return (0, _utility_types.assertUnreachable)(diffOutcome);
  }
};