"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildEsQueryParams = buildEsQueryParams;
exports.checkIfRemediationExists = void 0;
exports.createDatastream = createDatastream;
exports.createPipeline = createPipeline;
exports.generateInsightId = generateInsightId;
exports.generateTrustedAppsFilter = void 0;
exports.getUniqueInsights = getUniqueInsights;
exports.groupEndpointIdsByOS = groupEndpointIdsByOS;
var _crypto = require("crypto");
var _lodash = require("lodash");
var _dataStreamAdapter = require("@kbn/data-stream-adapter");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _constants = require("./constants");
var _field_map_configurations = require("./field_map_configurations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createDatastream(kibanaVersion) {
  const ds = new _dataStreamAdapter.DataStreamSpacesAdapter(_constants.DATA_STREAM_PREFIX, {
    kibanaVersion,
    totalFieldsLimit: _constants.TOTAL_FIELDS_LIMIT
  });
  ds.setComponentTemplate({
    name: _constants.COMPONENT_TEMPLATE_NAME,
    fieldMap: _field_map_configurations.securityWorkflowInsightsFieldMap
  });
  ds.setIndexTemplate({
    name: _constants.INDEX_TEMPLATE_NAME,
    componentTemplateRefs: [_constants.COMPONENT_TEMPLATE_NAME],
    template: {
      settings: {
        default_pipeline: _constants.INGEST_PIPELINE_NAME
      }
    },
    hidden: true
  });
  return ds;
}
async function createPipeline(esClient) {
  const response = await esClient.ingest.putPipeline({
    id: _constants.INGEST_PIPELINE_NAME,
    processors: [
      // requires @elastic/elasticsearch 8.16.0
      // {
      //   fingerprint: {
      //     fields: ['type', 'category', 'value', 'target.type', 'target.id'],
      //     target_field: '_id',
      //     method: 'SHA-256',
      //     if: 'ctx._id == null',
      //   },
      // },
    ],
    _meta: {
      managed: true
    }
  });
  return response.acknowledged;
}
const validKeys = new Set(['ids', 'categories', 'types', 'sourceTypes', 'sourceIds', 'targetTypes', 'targetIds', 'actionTypes']);
const paramFieldMap = {
  ids: '_id',
  sourceTypes: 'source.type',
  sourceIds: 'source.id',
  targetTypes: 'target.type',
  targetIds: 'target.ids',
  actionTypes: 'action.type'
};
const nestedKeys = new Set(['source', 'target', 'action']);
function buildEsQueryParams(searchParams) {
  return Object.entries(searchParams).reduce((acc, [k, v]) => {
    if (!validKeys.has(k)) {
      return acc;
    }
    const paramKey = (0, _lodash.get)(paramFieldMap, k, k);
    const topKey = paramKey.split('.')[0];
    if (nestedKeys.has(topKey)) {
      const nestedQuery = {
        nested: {
          path: topKey,
          query: {
            terms: {
              [paramKey]: v
            }
          }
        }
      };
      return [...acc, nestedQuery];
    }
    const next = {
      terms: {
        [paramKey]: v
      }
    };
    return [...acc, next];
  }, []);
}
async function groupEndpointIdsByOS(endpointIds, endpointMetadataService) {
  const metadata = await endpointMetadataService.getMetadataForEndpoints(endpointIds);
  return metadata.reduce((acc, m) => {
    const os = m.host.os.name.toLowerCase();
    if (!acc[os]) {
      acc[os] = [];
    }
    acc[os].push(m.agent.id);
    return acc;
  }, {});
}
function generateInsightId(insight) {
  const {
    type,
    category,
    value,
    target
  } = insight;
  const targetType = target.type;
  const targetIds = target.ids.join(',');
  const hash = (0, _crypto.createHash)('sha256');
  hash.update(type);
  hash.update(category);
  hash.update(value);
  hash.update(targetType);
  hash.update(targetIds);
  return hash.digest('hex');
}
function getUniqueInsights(insights) {
  const uniqueInsights = {};
  for (const insight of insights) {
    const id = generateInsightId(insight);
    if (!uniqueInsights[id]) {
      uniqueInsights[id] = insight;
    }
  }
  return Object.values(uniqueInsights);
}
const generateTrustedAppsFilter = insight => {
  var _insight$remediation$;
  return (_insight$remediation$ = insight.remediation.exception_list_items) === null || _insight$remediation$ === void 0 ? void 0 : _insight$remediation$.flatMap(item => item.entries.map(entry => {
    if (!('value' in entry)) return '';
    if (entry.field === 'process.executable.caseless') {
      return `exception-list-agnostic.attributes.entries.value:"${entry.value}"`;
    }
    if (entry.field === 'process.code_signature' || entry.field === 'process.Ext.code_signature' && typeof entry.value === 'string') {
      const sanitizedValue = entry.value.replace(/[)(<>}{":\\]/gm, '\\$&').replace(/\s/gm, '*');
      return `exception-list-agnostic.attributes.entries.entries.value:(*${sanitizedValue}*)`;
    }
    return '';
  })).filter(Boolean).join(' AND ');
};
exports.generateTrustedAppsFilter = generateTrustedAppsFilter;
const checkIfRemediationExists = async ({
  insight,
  exceptionListsClient
}) => {
  if (insight.type !== _elasticAssistantCommon.DefendInsightType.Enum.incompatible_antivirus) {
    return false;
  }
  const filter = generateTrustedAppsFilter(insight);
  if (!filter) return false;
  const response = await exceptionListsClient.findExceptionListItem({
    listId: 'endpoint_trusted_apps',
    page: 1,
    perPage: 1,
    namespaceType: 'agnostic',
    filter,
    sortField: 'created_at',
    sortOrder: 'desc'
  });
  return !!(response !== null && response !== void 0 && response.total) && response.total > 0;
};
exports.checkIfRemediationExists = checkIfRemediationExists;