"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ManifestManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _semver = _interopRequireDefault(require("semver"));
var _lodash = require("lodash");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _keys = require("@kbn/security-solution-features/keys");
var _std = require("@kbn/std");
var _unified_manifest_client = require("../unified_manifest_client");
var _stringify = require("../../../utils/stringify");
var _queue_processor = require("../../../utils/queue_processor");
var _manifest = require("../../../../../common/endpoint/schema/manifest");
var _artifacts = require("../../../lib/artifacts");
var _artifacts2 = require("../../../schemas/artifacts");
var _manifest_client = require("../manifest_client");
var _errors = require("../errors");
var _utils = require("../../../utils");
var _errors2 = require("../../../../../common/endpoint/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const iterateArtifactsBuildResult = (result, callback) => {
  for (const artifact of result.defaultArtifacts) {
    callback(artifact);
  }
  for (const policyId of Object.keys(result.policySpecificArtifacts)) {
    for (const artifact of result.policySpecificArtifacts[policyId]) {
      callback(artifact, policyId);
    }
  }
};
const getArtifactIds = manifest => [...Object.keys(manifest.artifacts)].map(key => `${key}-${manifest.artifacts[key].decoded_sha256}`);
const manifestsEqual = (manifest1, manifest2) => (0, _lodash.isEqual)(new Set(getArtifactIds(manifest1)), new Set(getArtifactIds(manifest2)));
class ManifestManager {
  constructor(context) {
    (0, _defineProperty2.default)(this, "artifactClient", void 0);
    (0, _defineProperty2.default)(this, "exceptionListClient", void 0);
    (0, _defineProperty2.default)(this, "packagePolicyService", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "schemaVersion", void 0);
    (0, _defineProperty2.default)(this, "experimentalFeatures", void 0);
    (0, _defineProperty2.default)(this, "cachedExceptionsListsByOs", void 0);
    (0, _defineProperty2.default)(this, "packagerTaskPackagePolicyUpdateBatchSize", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "productFeaturesService", void 0);
    this.artifactClient = context.artifactClient;
    this.exceptionListClient = context.exceptionListClient;
    this.packagePolicyService = context.packagePolicyService;
    this.savedObjectsClient = context.savedObjectsClient;
    this.logger = context.logger;
    this.schemaVersion = 'v1';
    this.experimentalFeatures = context.experimentalFeatures;
    this.cachedExceptionsListsByOs = new Map();
    this.packagerTaskPackagePolicyUpdateBatchSize = context.packagerTaskPackagePolicyUpdateBatchSize;
    this.esClient = context.esClient;
    this.productFeaturesService = context.productFeaturesService;
  }

  /**
   * Gets a ManifestClient for this manager's schemaVersion.
   *
   * @returns {ManifestClient} A ManifestClient scoped to the appropriate schemaVersion.
   */
  getManifestClient() {
    return new _manifest_client.ManifestClient(this.savedObjectsClient, this.schemaVersion);
  }

  /**
   * Search or get exceptions from the cached map by listId and OS and filter those by policyId/global
   */
  async getCachedExceptions({
    elClient,
    listId,
    os,
    policyId,
    schemaVersion,
    exceptionItemDecorator
  }) {
    if (!this.cachedExceptionsListsByOs.has(`${listId}-${os}`)) {
      let itemsByListId = [];
      // endpointHostIsolationExceptions includes full CRUD support for Host Isolation Exceptions
      // endpointArtifactManagement includes full CRUD support for all other exception lists + RD support for Host Isolation Exceptions
      // If there are host isolation exceptions in place but there is a downgrade scenario, those shouldn't be taken into account when generating artifacts.
      if (listId === _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.hostIsolationExceptions.id && this.productFeaturesService.isEnabled(_keys.ProductFeatureKey.endpointHostIsolationExceptions) || listId !== _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.hostIsolationExceptions.id && this.productFeaturesService.isEnabled(_keys.ProductFeatureKey.endpointArtifactManagement)) {
        itemsByListId = await (0, _artifacts.getAllItemsFromEndpointExceptionList)({
          elClient,
          os,
          listId
        });
        if (exceptionItemDecorator) {
          itemsByListId = itemsByListId.map(exceptionItemDecorator);
        }
      }
      this.cachedExceptionsListsByOs.set(`${listId}-${os}`, itemsByListId);
    }
    const allExceptionsByListId = this.cachedExceptionsListsByOs.get(`${listId}-${os}`);
    if (!allExceptionsByListId) {
      throw new _errors.InvalidInternalManifestError(`Error getting exceptions for ${listId}-${os}`);
    }
    const filter = exception => policyId ? exception.tags.includes('policy:all') || exception.tags.includes(`policy:${policyId}`) : exception.tags.includes('policy:all');
    const exceptions = listId === _securitysolutionListConstants.ENDPOINT_LIST_ID ? allExceptionsByListId : allExceptionsByListId.filter(filter);
    return (0, _artifacts.convertExceptionsToEndpointFormat)(exceptions, schemaVersion, this.experimentalFeatures);
  }

  /**
   * Builds an artifact (one per supported OS) based on the current state of the
   * artifacts list (Trusted Apps, Host Iso. Exceptions, Event Filters, Blocklists)
   * (which uses the `exception-list-agnostic` SO type)
   */
  async buildArtifactsForOs({
    listId,
    name,
    os,
    policyId,
    exceptionItemDecorator
  }) {
    return (0, _artifacts.buildArtifact)(await this.getCachedExceptions({
      elClient: this.exceptionListClient,
      schemaVersion: this.schemaVersion,
      os,
      policyId,
      listId,
      exceptionItemDecorator
    }), this.schemaVersion, os, name);
  }

  /**
   * Builds an artifact (by policy) based on the current state of the
   * artifacts list (Trusted Apps, Host Iso. Exceptions, Event Filters, Blocklists)
   * (which uses the `exception-list-agnostic` SO type)
   */
  async buildArtifactsByPolicy(allPolicyIds, supportedOSs, osOptions) {
    const policySpecificArtifacts = {};
    for (const policyId of allPolicyIds) for (const os of supportedOSs) {
      policySpecificArtifacts[policyId] = policySpecificArtifacts[policyId] || [];
      policySpecificArtifacts[policyId].push(await this.buildArtifactsForOs({
        os,
        policyId,
        ...osOptions
      }));
    }
    return policySpecificArtifacts;
  }

  /**
   * Builds an array of artifacts (one per supported OS) based on the current
   * state of exception-list-agnostic SOs.
   *
   * @returns {Promise<InternalArtifactCompleteSchema[]>} An array of uncompressed artifacts built from exception-list-agnostic SOs.
   * @throws Throws/rejects if there are errors building the list.
   */
  async buildExceptionListArtifacts(allPolicyIds) {
    const defaultArtifacts = [];
    const policySpecificArtifacts = {};
    const decorateWildcardOnlyExceptionItem = item => {
      const isWildcardOnly = item.entries.every(({
        type
      }) => type === 'wildcard');

      // add `event.module=endpoint` to make endpoints older than 8.2 work when only `wildcard` is used
      if (isWildcardOnly) {
        item.entries.push({
          type: 'match',
          operator: 'included',
          field: 'event.module',
          value: 'endpoint'
        });
      }
      return item;
    };
    const buildArtifactsForOsOptions = {
      listId: _securitysolutionListConstants.ENDPOINT_LIST_ID,
      name: _artifacts.ArtifactConstants.GLOBAL_ALLOWLIST_NAME,
      exceptionItemDecorator: decorateWildcardOnlyExceptionItem
    };
    for (const os of _artifacts.ArtifactConstants.SUPPORTED_OPERATING_SYSTEMS) {
      defaultArtifacts.push(await this.buildArtifactsForOs({
        os,
        ...buildArtifactsForOsOptions
      }));
    }
    allPolicyIds.forEach(policyId => {
      policySpecificArtifacts[policyId] = defaultArtifacts;
    });
    return {
      defaultArtifacts,
      policySpecificArtifacts
    };
  }

  /**
   * Builds an array of artifacts (one per supported OS) based on the current state of the
   * Trusted Apps list (which uses the `exception-list-agnostic` SO type)
   */
  async buildTrustedAppsArtifacts(allPolicyIds) {
    const defaultArtifacts = [];
    const buildArtifactsForOsOptions = {
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.trustedApps.id,
      name: _artifacts.ArtifactConstants.GLOBAL_TRUSTED_APPS_NAME
    };
    for (const os of _artifacts.ArtifactConstants.SUPPORTED_TRUSTED_APPS_OPERATING_SYSTEMS) {
      defaultArtifacts.push(await this.buildArtifactsForOs({
        os,
        ...buildArtifactsForOsOptions
      }));
    }
    const policySpecificArtifacts = await this.buildArtifactsByPolicy(allPolicyIds, _artifacts.ArtifactConstants.SUPPORTED_TRUSTED_APPS_OPERATING_SYSTEMS, buildArtifactsForOsOptions);
    return {
      defaultArtifacts,
      policySpecificArtifacts
    };
  }

  /**
   * Builds an array of endpoint event filters (one per supported OS) based on the current state of the
   * Event Filters list
   * @protected
   */
  async buildEventFiltersArtifacts(allPolicyIds) {
    const defaultArtifacts = [];
    const buildArtifactsForOsOptions = {
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id,
      name: _artifacts.ArtifactConstants.GLOBAL_EVENT_FILTERS_NAME
    };
    for (const os of _artifacts.ArtifactConstants.SUPPORTED_EVENT_FILTERS_OPERATING_SYSTEMS) {
      defaultArtifacts.push(await this.buildArtifactsForOs({
        os,
        ...buildArtifactsForOsOptions
      }));
    }
    const policySpecificArtifacts = await this.buildArtifactsByPolicy(allPolicyIds, _artifacts.ArtifactConstants.SUPPORTED_EVENT_FILTERS_OPERATING_SYSTEMS, buildArtifactsForOsOptions);
    return {
      defaultArtifacts,
      policySpecificArtifacts
    };
  }

  /**
   * Builds an array of Blocklist entries (one per supported OS) based on the current state of the
   * Blocklist list
   * @protected
   */
  async buildBlocklistArtifacts(allPolicyIds) {
    const defaultArtifacts = [];
    const buildArtifactsForOsOptions = {
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.blocklists.id,
      name: _artifacts.ArtifactConstants.GLOBAL_BLOCKLISTS_NAME
    };
    for (const os of _artifacts.ArtifactConstants.SUPPORTED_BLOCKLISTS_OPERATING_SYSTEMS) {
      defaultArtifacts.push(await this.buildArtifactsForOs({
        os,
        ...buildArtifactsForOsOptions
      }));
    }
    const policySpecificArtifacts = await this.buildArtifactsByPolicy(allPolicyIds, _artifacts.ArtifactConstants.SUPPORTED_BLOCKLISTS_OPERATING_SYSTEMS, buildArtifactsForOsOptions);
    return {
      defaultArtifacts,
      policySpecificArtifacts
    };
  }

  /**
   * Builds an array of endpoint host isolation exception (one per supported OS) based on the current state of the
   * Host Isolation Exception List
   * @returns
   */

  async buildHostIsolationExceptionsArtifacts(allPolicyIds) {
    const defaultArtifacts = [];
    const buildArtifactsForOsOptions = {
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.hostIsolationExceptions.id,
      name: _artifacts.ArtifactConstants.GLOBAL_HOST_ISOLATION_EXCEPTIONS_NAME
    };
    for (const os of _artifacts.ArtifactConstants.SUPPORTED_HOST_ISOLATION_EXCEPTIONS_OPERATING_SYSTEMS) {
      defaultArtifacts.push(await this.buildArtifactsForOs({
        os,
        ...buildArtifactsForOsOptions
      }));
    }
    const policySpecificArtifacts = await this.buildArtifactsByPolicy(allPolicyIds, _artifacts.ArtifactConstants.SUPPORTED_HOST_ISOLATION_EXCEPTIONS_OPERATING_SYSTEMS, buildArtifactsForOsOptions);
    return {
      defaultArtifacts,
      policySpecificArtifacts
    };
  }

  /**
   * Writes new artifact to Fleet
   *
   * @param artifacts An InternalArtifactCompleteSchema array representing the artifacts.
   * @param newManifest A Manifest representing the new manifest
   * @returns {Promise<Error[]>} Any errors encountered.
   */
  async pushArtifacts(artifacts, newManifest) {
    var _fleetArtifacts$lengt;
    const errors = [];
    const artifactsToCreate = [];
    for (const artifact of artifacts) {
      if (_artifacts2.internalArtifactCompleteSchema.is(artifact)) {
        artifactsToCreate.push(artifact);
      } else {
        errors.push(new _errors2.EndpointError(`Incomplete artifact: ${(0, _artifacts.getArtifactId)(artifact)}`, artifact));
      }
    }
    if (artifactsToCreate.length === 0) {
      return errors;
    }
    this.logger.debug(`Creating [${artifactsToCreate.length}] artifacts`);
    const {
      artifacts: fleetArtifacts,
      errors: createErrors
    } = await this.artifactClient.bulkCreateArtifacts(artifactsToCreate);
    this.logger.info(`Count of artifacts created: ${(_fleetArtifacts$lengt = fleetArtifacts === null || fleetArtifacts === void 0 ? void 0 : fleetArtifacts.length) !== null && _fleetArtifacts$lengt !== void 0 ? _fleetArtifacts$lengt : 0}`);
    if (createErrors) {
      errors.push(...createErrors);
    }
    const newArtifactsAddedToManifest = [];
    const artifactsNotCreated = [];
    if (fleetArtifacts) {
      const fleetArtifactsByIdentifier = {};
      fleetArtifacts.forEach(fleetArtifact => {
        fleetArtifactsByIdentifier[(0, _artifacts.getArtifactId)(fleetArtifact)] = fleetArtifact;
      });
      artifactsToCreate.forEach(artifact => {
        const artifactId = (0, _artifacts.getArtifactId)(artifact);
        const fleetArtifact = fleetArtifactsByIdentifier[artifactId];
        if (!fleetArtifact) {
          artifactsNotCreated.push(artifactId);
          return;
        }
        newManifest.replaceArtifact(fleetArtifact);
        newArtifactsAddedToManifest.push(artifactId);
      });
    }
    if (artifactsNotCreated.length) {
      this.logger.debug(`A total of [${artifactsNotCreated.length}] artifacts were not created. Prior version of the artifact will remain in manifest.\n${artifactsNotCreated.join('\n')}`);
    }
    if (newArtifactsAddedToManifest.length !== 0) {
      this.logger.debug(`Newly created artifacts added to the manifest:\n${newArtifactsAddedToManifest.join('\n')}`);
    }
    return errors;
  }

  /**
   * Deletes outdated artifact SOs.
   *
   * @param artifactIds The IDs of the artifact to delete..
   * @returns {Promise<Error[]>} Any errors encountered.
   */
  async deleteArtifacts(artifactIds) {
    try {
      if ((0, _lodash.isEmpty)(artifactIds)) {
        return [];
      }
      const errors = await this.artifactClient.bulkDeleteArtifacts(artifactIds);
      if (!(0, _lodash.isEmpty)(errors)) {
        return errors;
      }
      this.logger.info(`Count of cleaned up artifacts: ${artifactIds.length}`);
      if (artifactIds.length !== 0) {
        this.logger.debug(`Deleted artifacts from cleanup:\n${artifactIds.join('\n  ')}`);
      }
      return [];
    } catch (err) {
      this.logger.error(`Attempted to delete [${artifactIds.length}] outdated artifacts failed with: ${err.message}\n${err.stack}`);
      return [err];
    }
  }

  /**
   * Returns the last computed manifest based on the state of the user-artifact-manifest SO. If no
   * artifacts have been created yet (ex. no Endpoint policies are in use), then method return `null`
   *
   * @returns {Promise<Manifest | null>} The last computed manifest, or null if does not exist.
   * @throws Throws/rejects if there is an unexpected error retrieving the manifest.
   */
  async getLastComputedManifest() {
    try {
      let manifestSo;
      if (this.experimentalFeatures.unifiedManifestEnabled) {
        const unifiedManifestsSo = await this.getAllUnifiedManifestsSO();
        // On first run, there will be no existing Unified Manifests SO, so we need to copy the semanticVersion from the legacy manifest
        // This is to ensure that the first Unified Manifest created has the same semanticVersion as the legacy manifest and is not too far
        // behind for package policy to pick it up.
        if (unifiedManifestsSo.length === 0) {
          var _legacyManifestSo$att;
          const legacyManifestSo = await this.getManifestClient().getManifest();
          const legacySemanticVersion = legacyManifestSo === null || legacyManifestSo === void 0 ? void 0 : (_legacyManifestSo$att = legacyManifestSo.attributes) === null || _legacyManifestSo$att === void 0 ? void 0 : _legacyManifestSo$att.semanticVersion;
          manifestSo = this.transformUnifiedManifestSOtoLegacyManifestSO(unifiedManifestsSo, legacySemanticVersion);
        } else {
          manifestSo = this.transformUnifiedManifestSOtoLegacyManifestSO(unifiedManifestsSo);
        }
      } else {
        manifestSo = await this.getManifestClient().getManifest();
      }
      if (manifestSo.version === undefined) {
        throw new _errors.InvalidInternalManifestError('Internal Manifest map SavedObject is missing version', manifestSo);
      }
      const manifest = new _artifacts.Manifest({
        schemaVersion: this.schemaVersion,
        semanticVersion: manifestSo.attributes.semanticVersion,
        soVersion: manifestSo.version
      });
      const fleetArtifacts = await this.listAllArtifacts();
      const fleetArtifactsById = (0, _lodash.keyBy)(fleetArtifacts, artifact => (0, _artifacts.getArtifactId)(artifact));
      const invalidArtifactIds = [];

      // Ensure that all artifacts currently defined in the Manifest have a valid artifact in fleet,
      // and remove any that does not have an actual artifact from the manifest
      for (const entry of manifestSo.attributes.artifacts) {
        const artifact = fleetArtifactsById[entry.artifactId];
        if (!artifact) {
          invalidArtifactIds.push(entry.artifactId);
        } else {
          manifest.addEntry(artifact, entry.policyId);
        }
      }
      if (invalidArtifactIds.length) {
        this.logger.warn(`Missing artifacts detected! Internal artifact manifest (SavedObject version [${manifestSo.version}]) references [${invalidArtifactIds.length}] artifact IDs that don't exist.\nFirst 10 below (run with logging set to 'debug' to see all):\n${invalidArtifactIds.slice(0, 10).join('\n')}`);
        this.logger.debug(`Artifact ID references that are missing:\n${(0, _stringify.stringify)(invalidArtifactIds)}`);
      }
      return manifest;
    } catch (error) {
      if (!error.output || error.output.statusCode !== 404) {
        throw (0, _utils.wrapErrorIfNeeded)(error);
      }
      return null;
    }
  }

  /**
   * creates a new default Manifest
   */
  static createDefaultManifest(schemaVersion) {
    return _artifacts.Manifest.getDefault(schemaVersion);
  }

  /**
   * Builds a new manifest based on the current user exception list.
   *
   * @param baselineManifest A baseline manifest to use for initializing pre-existing artifacts.
   * @returns {Promise<Manifest>} A new Manifest object representing the current exception list.
   */
  async buildNewManifest(baselineManifest = ManifestManager.createDefaultManifest(this.schemaVersion)) {
    const allPolicyIds = await this.listEndpointPolicyIds();
    const results = await Promise.all([this.buildExceptionListArtifacts(allPolicyIds), this.buildTrustedAppsArtifacts(allPolicyIds), this.buildEventFiltersArtifacts(allPolicyIds), this.buildHostIsolationExceptionsArtifacts(allPolicyIds), this.buildBlocklistArtifacts(allPolicyIds)]);

    // Clear cache as the ManifestManager instance is reused on every run.
    this.cachedExceptionsListsByOs.clear();
    const manifest = new _artifacts.Manifest({
      schemaVersion: this.schemaVersion,
      semanticVersion: baselineManifest.getSemanticVersion(),
      soVersion: baselineManifest.getSavedObjectVersion()
    });
    for (const result of results) {
      iterateArtifactsBuildResult(result, (artifact, policyId) => {
        manifest.addEntry(baselineManifest.getArtifact((0, _artifacts.getArtifactId)(artifact)) || artifact, policyId);
      });
    }
    return manifest;
  }

  /**
   * Dispatches the manifest by writing it to the endpoint package policy, if different
   * from the manifest already in the config.
   *
   * @param manifest The Manifest to dispatch.
   * @returns {Promise<Error[]>} Any errors encountered.
   */
  async tryDispatch(manifest) {
    const errors = [];
    const updatedPolicies = [];
    const unChangedPolicies = [];
    const manifestVersion = manifest.getSemanticVersion();
    const execId = Math.random().toString(32).substring(3, 8);
    const policyUpdateBatchProcessor = new _queue_processor.QueueProcessor({
      batchSize: this.packagerTaskPackagePolicyUpdateBatchSize,
      logger: this.logger,
      key: `tryDispatch.${execId}`,
      batchHandler: async ({
        data: currentBatch
      }) => {
        const response = await this.packagePolicyService.bulkUpdate(this.savedObjectsClient, this.esClient, currentBatch);
        if (!(0, _lodash.isEmpty)(response.failedPolicies)) {
          errors.push(...response.failedPolicies.map(failedPolicy => {
            if (failedPolicy.error instanceof Error) {
              return failedPolicy.error;
            } else {
              return new Error(failedPolicy.error.message);
            }
          }));
        }
        if (response.updatedPolicies) {
          updatedPolicies.push(...response.updatedPolicies.map(policy => {
            return `[${policy.id}][${policy.name}] updated with manifest version: [${manifestVersion}]`;
          }));
        }
      }
    });
    for await (const policies of await this.fetchAllPolicies()) {
      for (const packagePolicy of policies) {
        const {
          id,
          name
        } = packagePolicy;
        if (packagePolicy.inputs.length > 0 && packagePolicy.inputs[0].config !== undefined) {
          var _packagePolicy$inputs;
          const oldManifest = (_packagePolicy$inputs = packagePolicy.inputs[0].config.artifact_manifest) !== null && _packagePolicy$inputs !== void 0 ? _packagePolicy$inputs : {
            value: {}
          };
          const newManifestVersion = manifest.getSemanticVersion();
          if (_semver.default.gt(newManifestVersion, oldManifest.value.manifest_version)) {
            const serializedManifest = manifest.toPackagePolicyManifest(id);
            if (!_manifest.manifestDispatchSchema.is(serializedManifest)) {
              errors.push(new _errors2.EndpointError(`Invalid manifest for policy ${id}`, serializedManifest));
            } else if (!manifestsEqual(serializedManifest, oldManifest.value)) {
              packagePolicy.inputs[0].config.artifact_manifest = {
                value: serializedManifest
              };
              policyUpdateBatchProcessor.addToQueue(packagePolicy);
            } else {
              unChangedPolicies.push(`[${id}][${name}] No change in manifest content`);
            }
          } else {
            unChangedPolicies.push(`[${id}][${name}] No change in manifest version`);
          }
        } else {
          errors.push(new _errors2.EndpointError(`Package Policy ${id} has no 'inputs[0].config'`, packagePolicy));
        }
      }
    }
    await policyUpdateBatchProcessor.complete();
    this.logger.debug(`Processed [${updatedPolicies.length + unChangedPolicies.length}] Policies: updated: [${updatedPolicies.length}], un-changed: [${unChangedPolicies.length}]`);
    if (updatedPolicies.length) {
      this.logger.debug(`Updated Policies:\n  ${updatedPolicies.join('\n  ')}`);
    }
    if (unChangedPolicies.length) {
      this.logger.debug(`Un-changed Policies:\n  ${unChangedPolicies.join('\n  ')}`);
    }
    return errors;
  }

  /**
   * Commits a manifest to indicate that a new version has been computed.
   *
   * @param manifest The Manifest to commit.
   * @returns {Promise<Error | null>} An error, if encountered, or null.
   */
  async commit(manifest) {
    const manifestSo = manifest.toSavedObject();
    if (this.experimentalFeatures.unifiedManifestEnabled) {
      await this.commitUnified(manifestSo);
    } else {
      const manifestClient = this.getManifestClient();
      const version = manifest.getSavedObjectVersion();
      if (version == null) {
        await manifestClient.createManifest(manifestSo);
      } else {
        await manifestClient.updateManifest(manifestSo, {
          version
        });
      }
      this.logger.debug(`Committed manifest ${manifest.getSemanticVersion()}`);
    }
  }
  fetchAllPolicies() {
    return this.packagePolicyService.fetchAllItems(this.savedObjectsClient, {
      kuery: 'ingest-package-policies.package.name:endpoint'
    });
  }
  async listEndpointPolicyIds() {
    const allPolicyIds = [];
    const idFetcher = await this.packagePolicyService.fetchAllItemIds(this.savedObjectsClient, {
      kuery: 'ingest-package-policies.package.name:endpoint'
    });
    for await (const itemIds of idFetcher) {
      allPolicyIds.push(...itemIds);
    }
    this.logger.debug(`Retrieved [${allPolicyIds.length}] endpoint integration policy IDs`);
    return allPolicyIds;
  }
  getArtifactsClient() {
    return this.artifactClient;
  }

  /**
   * Retrieves all .fleet-artifacts for endpoint package
   * @returns Artifact[]
   */
  async listAllArtifacts() {
    const fleetArtifacts = [];
    let total = 0;
    for await (const artifacts of this.artifactClient.fetchAll()) {
      fleetArtifacts.push(...artifacts);
      total += artifacts.length;
    }
    this.logger.debug(`Count of current stored artifacts: ${total}`);
    return fleetArtifacts;
  }

  /**
   * Pulls in all artifacts from Fleet and checks to ensure they are all being referenced
   * by the Manifest. If any are found to not be in the current Manifest (orphan), they
   * are cleaned up (deleted)
   */
  async cleanup(manifest) {
    const badArtifactIds = [];
    const errors = [];
    const artifactDeletionProcess = new _queue_processor.QueueProcessor({
      batchSize: this.packagerTaskPackagePolicyUpdateBatchSize,
      logger: this.logger,
      key: 'cleanup',
      batchHandler: async ({
        batch,
        data
      }) => {
        const deleteErrors = await this.artifactClient.bulkDeleteArtifacts(data);
        badArtifactIds.push(...data);
        if (deleteErrors.length) {
          errors.push(`Delete batch #[${batch}] with [${data.length}] items:\n${(0, _stringify.stringify)(deleteErrors)}`);
        }
      }
    });
    const validArtifactIds = manifest.getAllArtifacts().map(artifact => (0, _artifacts.getArtifactId)(artifact));
    for await (const artifacts of this.artifactClient.fetchAll()) {
      for (const artifact of artifacts) {
        const artifactId = (0, _artifacts.getArtifactId)(artifact);
        const isArtifactInManifest = validArtifactIds.includes(artifactId);
        if (!isArtifactInManifest) {
          artifactDeletionProcess.addToQueue(artifactId);
        }
      }
    }
    await artifactDeletionProcess.complete();
    if (errors.length > 0) {
      this.logger.error(`The following errors were encountered while attempting to delete [${badArtifactIds.length}] orphaned artifacts:\n${(0, _stringify.stringify)(errors)}`);
    } else if (badArtifactIds.length > 0) {
      this.logger.debug(`Count of orphan artifacts cleaned up: ${badArtifactIds.length}\n${(0, _stringify.stringify)(badArtifactIds)}`);
    }
  }

  /**
   * Unified Manifest methods
   */

  setNewSemanticVersion(semanticVersion) {
    const newSemanticVersion = _semver.default.inc(semanticVersion, 'patch');
    if (!_semver.default.valid(newSemanticVersion)) {
      throw new Error(`Invalid semver: ${newSemanticVersion}`);
    }
    return newSemanticVersion;
  }
  getUnifiedManifestClient() {
    return new _unified_manifest_client.UnifiedManifestClient(this.savedObjectsClient);
  }
  async getAllUnifiedManifestsSO() {
    return this.getUnifiedManifestClient().getAllUnifiedManifests();
  }
  transformUnifiedManifestSOtoLegacyManifestSO(unifiedManifestsSo, semanticVersion) {
    var _globalUnifiedManifes, _ref;
    const globalUnifiedManifest = unifiedManifestsSo.find(a => a.policyId === '.global');
    return {
      version: 'WzQ3NzAsMV0=',
      // version is hardcoded since it was used only to determine whether to create a new manifest or update an existing one
      attributes: {
        artifacts: [...((_globalUnifiedManifes = globalUnifiedManifest === null || globalUnifiedManifest === void 0 ? void 0 : globalUnifiedManifest.artifactIds.map(artifactId => ({
          artifactId,
          policyId: undefined
        }))) !== null && _globalUnifiedManifes !== void 0 ? _globalUnifiedManifes : []), ...unifiedManifestsSo.reduce((acc, unifiedManifest) => {
          if (unifiedManifest.policyId === '.global') {
            return acc;
          }
          acc.push(...unifiedManifest.artifactIds.map(artifactId => ({
            policyId: unifiedManifest.policyId,
            artifactId
          })));
          return acc;
        }, [])],
        semanticVersion: (_ref = semanticVersion || (globalUnifiedManifest === null || globalUnifiedManifest === void 0 ? void 0 : globalUnifiedManifest.semanticVersion)) !== null && _ref !== void 0 ? _ref : '1.0.0',
        schemaVersion: this.schemaVersion
      }
    };
  }
  transformLegacyManifestSOtoUnifiedManifestSO(manifestSo, unifiedManifestsSo) {
    const manifestObject = manifestSo.artifacts.reduce((acc, {
      artifactId,
      policyId = '.global'
    }) => {
      const existingPolicy = acc[policyId];
      if (existingPolicy) {
        existingPolicy.artifactIds.push(artifactId);
      } else {
        var _ref2;
        const existingUnifiedManifestSo = unifiedManifestsSo.find(item => item.policyId === policyId);

        // On first run, there will be no existing Unified Manifests SO, so we need to copy the semanticVersion from the legacy manifest
        // This is to ensure that the first Unified Manifest created has the same semanticVersion as the legacy manifest and is not too far
        // behind for package policy to pick it up.
        const semanticVersion = (_ref2 = policyId === '.global' && !unifiedManifestsSo.length ? manifestSo === null || manifestSo === void 0 ? void 0 : manifestSo.semanticVersion : existingUnifiedManifestSo === null || existingUnifiedManifestSo === void 0 ? void 0 : existingUnifiedManifestSo.semanticVersion) !== null && _ref2 !== void 0 ? _ref2 : '1.0.0';
        acc[policyId] = {
          policyId,
          artifactIds: [artifactId],
          semanticVersion,
          id: existingUnifiedManifestSo === null || existingUnifiedManifestSo === void 0 ? void 0 : existingUnifiedManifestSo.id
        };
      }
      return acc;
    }, {});
    return Object.values(manifestObject);
  }
  prepareUnifiedManifestsSOUpdates(unifiedManifestsSo, existingUnifiedManifestsSo) {
    const existingManifestsObj = {};
    existingUnifiedManifestsSo.forEach(manifest => {
      existingManifestsObj[manifest.id] = manifest;
    });
    const {
      unifiedManifestsToUpdate,
      unifiedManifestsToCreate
    } = unifiedManifestsSo.reduce((acc, unifiedManifest) => {
      if (unifiedManifest.id !== undefined) {
        // Manifest with id exists in SO, check if it needs to be updated
        const existingUnifiedManifest = existingManifestsObj[unifiedManifest.id];
        // Update SO if the artifactIds changed.
        if (!(0, _lodash.isEqual)(existingUnifiedManifest.artifactIds, unifiedManifest.artifactIds)) {
          acc.unifiedManifestsToUpdate.push({
            ...unifiedManifest,
            semanticVersion: this.setNewSemanticVersion(unifiedManifest.semanticVersion),
            version: existingUnifiedManifest.version
          });
        }
      } else {
        // Manifest with id does not exist in SO, create new SO
        acc.unifiedManifestsToCreate.push(unifiedManifest);
      }
      return acc;
    }, {
      unifiedManifestsToUpdate: [],
      unifiedManifestsToCreate: []
    });
    const unifiedManifestsToDelete = existingUnifiedManifestsSo.reduce((acc, {
      policyId,
      id
    }) => {
      const existingPolicy = unifiedManifestsSo.find(item => item.policyId === policyId);
      if (!existingPolicy) {
        acc.push(id);
      }
      return acc;
    }, []);
    return {
      unifiedManifestsToUpdate,
      unifiedManifestsToCreate,
      unifiedManifestsToDelete
    };
  }
  async bumpGlobalUnifiedManifestVersion() {
    var _globalUnifiedManifes2;
    const globalUnifiedManifestSO = await this.getUnifiedManifestClient().getUnifiedManifestByPolicyId('.global');
    if (!(globalUnifiedManifestSO !== null && globalUnifiedManifestSO !== void 0 && (_globalUnifiedManifes2 = globalUnifiedManifestSO.saved_objects) !== null && _globalUnifiedManifes2 !== void 0 && _globalUnifiedManifes2.length)) {
      this.logger.warn('No Global Unified Manifest found to bump version');
      return;
    }
    const globalUnifiedManifest = globalUnifiedManifestSO.saved_objects[0];
    const newSemanticVersion = this.setNewSemanticVersion(globalUnifiedManifest.attributes.semanticVersion) || '1.0.0';
    await this.getUnifiedManifestClient().updateUnifiedManifest({
      ...globalUnifiedManifest.attributes,
      id: globalUnifiedManifest.id,
      semanticVersion: newSemanticVersion
    });
  }
  async commitUnified(manifestSo) {
    const existingUnifiedManifestsSo = await this.getAllUnifiedManifestsSO();
    const unifiedManifestSO = this.transformLegacyManifestSOtoUnifiedManifestSO(manifestSo, existingUnifiedManifestsSo);
    const {
      unifiedManifestsToUpdate,
      unifiedManifestsToCreate,
      unifiedManifestsToDelete
    } = this.prepareUnifiedManifestsSOUpdates(unifiedManifestSO, existingUnifiedManifestsSo);
    if (unifiedManifestsToCreate.length) {
      await (0, _std.asyncForEach)((0, _lodash.chunk)(unifiedManifestsToCreate, 100), async unifiedManifestsBatch => {
        await this.getUnifiedManifestClient().createUnifiedManifests(unifiedManifestsBatch);
      });
      this.logger.debug(`Created ${unifiedManifestsToCreate.length} unified manifests`);
    }
    if (unifiedManifestsToUpdate.length) {
      await (0, _std.asyncForEach)((0, _lodash.chunk)(unifiedManifestsToUpdate, 100), async unifiedManifestsBatch => {
        await this.getUnifiedManifestClient().updateUnifiedManifests(unifiedManifestsBatch);
      });
      this.logger.debug(`Updated ${unifiedManifestsToUpdate.length} unified manifests`);
    }
    if (unifiedManifestsToDelete.length) {
      await (0, _std.asyncForEach)((0, _lodash.chunk)(unifiedManifestsToDelete, 100), async unifiedManifestsBatch => {
        await this.getUnifiedManifestClient().deleteUnifiedManifestByIds(unifiedManifestsBatch);
      });
      this.logger.debug(`Deleted ${unifiedManifestsToDelete.length} unified manifests`);
    }
    if (unifiedManifestsToCreate.length || unifiedManifestsToUpdate.length || unifiedManifestsToDelete.length) {
      // If global manifest is not in the list of manifests to create or update, we need to bump its version
      // We use it to set schemaVersion of the legacy manifest we are going to create so that its being picked up when populating agent policy
      const hasGlobalManifest = [...unifiedManifestsToCreate, ...unifiedManifestsToUpdate].some(manifest => manifest.policyId === '.global');
      if (!hasGlobalManifest || unifiedManifestsToDelete.length) {
        await this.bumpGlobalUnifiedManifestVersion();
      }
    }
  }
}
exports.ManifestManager = ManifestManager;