"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getActionListByStatus = exports.getActionList = void 0;
var _fetch_action_requests = require("./utils/fetch_action_requests");
var _fetch_action_responses = require("./utils/fetch_action_responses");
var _constants = require("../../../../common/endpoint/constants");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _utils = require("./utils");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Similar to #getActionList but takes statuses filter options
 * Retrieve a list of all (at most 10k) Actions from index (`ActionDetails`)
 * filter out action details based on statuses filter options
 */
const getActionListByStatus = async ({
  agentTypes,
  commands,
  elasticAgentIds,
  esClient,
  endDate,
  logger,
  metadataService,
  page: _page,
  pageSize,
  startDate,
  statuses,
  userIds,
  unExpiredOnly = false,
  types,
  withOutputs
}) => {
  const size = pageSize !== null && pageSize !== void 0 ? pageSize : _constants.ENDPOINT_DEFAULT_PAGE_SIZE;
  const page = _page !== null && _page !== void 0 ? _page : 1;
  const {
    actionDetails: allActionDetails
  } = await getActionDetailsList({
    agentTypes,
    commands,
    elasticAgentIds,
    esClient,
    endDate,
    from: 0,
    logger,
    metadataService,
    size: _constants2.ACTIONS_SEARCH_PAGE_SIZE,
    startDate,
    userIds,
    unExpiredOnly,
    types,
    withOutputs
  });

  // filter out search results based on status filter options
  const actionDetailsByStatus = allActionDetails.filter(detail => statuses.includes(detail.status));
  return {
    page,
    pageSize: size,
    startDate,
    endDate,
    agentTypes,
    elasticAgentIds,
    userIds,
    commands,
    statuses,
    // for size 20 -> page 1: (0, 20), page 2: (20, 40) ...etc
    data: actionDetailsByStatus.slice((page - 1) * size, size * page),
    total: actionDetailsByStatus.length
  };
};

/**
 * Retrieve a list of Actions (`ActionDetails`)
 */
exports.getActionListByStatus = getActionListByStatus;
const getActionList = async ({
  agentTypes,
  commands,
  elasticAgentIds,
  esClient,
  endDate,
  logger,
  metadataService,
  page: _page,
  pageSize,
  startDate,
  userIds,
  unExpiredOnly = false,
  withOutputs,
  types
}) => {
  const size = pageSize !== null && pageSize !== void 0 ? pageSize : _constants.ENDPOINT_DEFAULT_PAGE_SIZE;
  const page = _page !== null && _page !== void 0 ? _page : 1;
  // # of hits to skip
  const from = (page - 1) * size;
  const {
    actionDetails,
    totalRecords
  } = await getActionDetailsList({
    agentTypes,
    commands,
    elasticAgentIds,
    esClient,
    endDate,
    from,
    logger,
    metadataService,
    size,
    startDate,
    userIds,
    unExpiredOnly,
    withOutputs,
    types
  });
  return {
    page,
    pageSize: size,
    startDate,
    endDate,
    agentTypes,
    elasticAgentIds,
    userIds,
    commands,
    statuses: undefined,
    data: actionDetails,
    total: totalRecords
  };
};
exports.getActionList = getActionList;
const getActionDetailsList = async ({
  agentTypes,
  commands,
  elasticAgentIds,
  esClient,
  endDate,
  from,
  logger,
  metadataService,
  size,
  startDate,
  userIds,
  unExpiredOnly,
  withOutputs,
  types
}) => {
  let actionRequests = [];
  let totalRecords = 0;
  try {
    const {
      data,
      total
    } = await (0, _fetch_action_requests.fetchActionRequests)({
      agentTypes,
      commands: commands,
      esClient,
      elasticAgentIds,
      startDate,
      endDate,
      from,
      size,
      userIds,
      unExpiredOnly,
      types: types,
      logger
    });
    actionRequests = data;
    totalRecords = total;
  } catch (error) {
    var _error$meta$meta$body, _error$meta, _error$meta$meta, _error$meta$meta$body2, _error$meta$meta$body3, _error$meta$meta$stat, _error$meta2, _error$meta2$meta;
    // all other errors
    const err = new _custom_http_request_error.CustomHttpRequestError((_error$meta$meta$body = (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$meta = _error$meta.meta) === null || _error$meta$meta === void 0 ? void 0 : (_error$meta$meta$body2 = _error$meta$meta.body) === null || _error$meta$meta$body2 === void 0 ? void 0 : (_error$meta$meta$body3 = _error$meta$meta$body2.error) === null || _error$meta$meta$body3 === void 0 ? void 0 : _error$meta$meta$body3.reason) !== null && _error$meta$meta$body !== void 0 ? _error$meta$meta$body : `Unknown error while fetching action requests (${error.message})`, (_error$meta$meta$stat = (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$meta = _error$meta2.meta) === null || _error$meta2$meta === void 0 ? void 0 : _error$meta2$meta.statusCode) !== null && _error$meta$meta$stat !== void 0 ? _error$meta$meta$stat : 500, error);
    logger.error(err);
    throw err;
  }
  if (!totalRecords) {
    return {
      actionDetails: [],
      totalRecords: 0
    };
  }
  const normalizedActionRequests = actionRequests.map(_utils.mapToNormalizedActionRequest);
  const agentIds = [];
  const actionReqIds = normalizedActionRequests.map(actionReq => {
    agentIds.push(...actionReq.agents);
    return actionReq.id;
  });
  let actionResponses;
  let agentsHostInfo;
  try {
    // get all responses for given action Ids and agent Ids
    // and get host metadata info with queried agents
    [actionResponses, agentsHostInfo] = await Promise.all([(0, _fetch_action_responses.fetchActionResponses)({
      esClient,
      agentIds: elasticAgentIds,
      actionIds: actionReqIds
    }), (0, _utils.getAgentHostNamesWithIds)({
      esClient,
      metadataService,
      agentIds
    })]);
  } catch (error) {
    var _error$meta$meta$body4, _error$meta3, _error$meta3$meta, _error$meta3$meta$bod, _error$meta3$meta$bod2, _error$meta$meta$stat2, _error$meta4, _error$meta4$meta;
    // all other errors
    const err = new _custom_http_request_error.CustomHttpRequestError((_error$meta$meta$body4 = (_error$meta3 = error.meta) === null || _error$meta3 === void 0 ? void 0 : (_error$meta3$meta = _error$meta3.meta) === null || _error$meta3$meta === void 0 ? void 0 : (_error$meta3$meta$bod = _error$meta3$meta.body) === null || _error$meta3$meta$bod === void 0 ? void 0 : (_error$meta3$meta$bod2 = _error$meta3$meta$bod.error) === null || _error$meta3$meta$bod2 === void 0 ? void 0 : _error$meta3$meta$bod2.reason) !== null && _error$meta$meta$body4 !== void 0 ? _error$meta$meta$body4 : `Unknown error while fetching action responses (${error.message})`, (_error$meta$meta$stat2 = (_error$meta4 = error.meta) === null || _error$meta4 === void 0 ? void 0 : (_error$meta4$meta = _error$meta4.meta) === null || _error$meta4$meta === void 0 ? void 0 : _error$meta4$meta.statusCode) !== null && _error$meta$meta$stat2 !== void 0 ? _error$meta$meta$stat2 : 500, error);
    logger.error(err);
    throw err;
  }
  const responsesByActionId = (0, _utils.mapResponsesByActionId)(actionResponses);
  const actionDetails = normalizedActionRequests.map(action => {
    var _responsesByActionId$;
    const actionRecord = (0, _utils.createActionDetailsRecord)(action, (_responsesByActionId$ = responsesByActionId[action.id]) !== null && _responsesByActionId$ !== void 0 ? _responsesByActionId$ : {
      fleetResponses: [],
      endpointResponses: []
    }, agentsHostInfo);
    if (withOutputs && !withOutputs.includes(action.id)) {
      delete actionRecord.outputs;
    }
    return actionRecord;
  });
  return {
    actionDetails,
    totalRecords
  };
};