"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerResponseActionRoutes = registerResponseActionRoutes;
var _i18n = require("@kbn/i18n");
var _endpoint = require("../../../../common/api/endpoint");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _stringify = require("../../utils/stringify");
var _error_handler = require("../error_handler");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _services = require("../../services");
var _constants2 = require("../../services/actions/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerResponseActionRoutes(router, endpointContext, docLinks) {
  const logger = endpointContext.logFactory.get('hostIsolation');

  /**
   * @deprecated use ISOLATE_HOST_ROUTE_V2 instead
   */
  router.versioned.post({
    access: 'public',
    path: _constants.ISOLATE_HOST_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.IsolateRouteRequestSchema
    },
    options: {
      deprecated: {
        documentationUrl: docLinks.links.securitySolution.legacyEndpointManagementApiDeprecations,
        severity: 'critical',
        message: _i18n.i18n.translate('xpack.securitySolution.deprecations.endpoint.response_actions.isolate', {
          defaultMessage: 'The "{path}" URL is deprecated and will be removed in the next major version.',
          values: {
            path: _constants.ISOLATE_HOST_ROUTE
          }
        }),
        reason: {
          type: 'remove'
        }
      }
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canIsolateHost']
  }, logger, redirectHandler(_constants.ISOLATE_HOST_ROUTE_V2)));

  /**
   * @deprecated use RELEASE_HOST_ROUTE instead
   */
  router.versioned.post({
    access: 'public',
    path: _constants.UNISOLATE_HOST_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UnisolateRouteRequestSchema
    },
    options: {
      deprecated: {
        documentationUrl: docLinks.links.securitySolution.legacyEndpointManagementApiDeprecations,
        severity: 'critical',
        message: _i18n.i18n.translate('xpack.securitySolution.deprecations.endpoint.response_actions.unisolate', {
          defaultMessage: 'The "{path}" URL is deprecated and will be removed in the next major version.',
          values: {
            path: _constants.UNISOLATE_HOST_ROUTE
          }
        }),
        reason: {
          type: 'remove'
        }
      }
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canUnIsolateHost']
  }, logger, redirectHandler(_constants.UNISOLATE_HOST_ROUTE_V2)));
  router.versioned.post({
    access: 'public',
    path: _constants.ISOLATE_HOST_ROUTE_V2,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.IsolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'isolate')));
  router.versioned.post({
    access: 'public',
    path: _constants.UNISOLATE_HOST_ROUTE_V2,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UnisolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canUnIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'unisolate')));
  router.versioned.post({
    access: 'public',
    path: _constants.KILL_PROCESS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.KillProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canKillProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'kill-process')));
  router.versioned.post({
    access: 'public',
    path: _constants.SUSPEND_PROCESS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.SuspendProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canSuspendProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'suspend-process')));
  router.versioned.post({
    access: 'public',
    path: _constants.GET_PROCESSES_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.GetProcessesRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canGetRunningProcesses']
  }, logger, responseActionRequestHandler(endpointContext, 'running-processes')));
  router.versioned.post({
    access: 'public',
    path: _constants.GET_FILE_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.EndpointActionGetFileSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'get-file')));
  router.versioned.post({
    access: 'public',
    path: _constants.EXECUTE_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.ExecuteActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteExecuteOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'execute')));
  router.versioned.post({
    access: 'public',
    path: _constants.UPLOAD_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true,
      body: {
        accepts: ['multipart/form-data'],
        output: 'stream',
        maxBytes: endpointContext.serverConfig.maxUploadResponseActionFileBytes
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UploadActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'upload')));
  router.versioned.post({
    access: 'public',
    path: _constants.SCAN_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.ScanActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteScanOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'scan')));
  router.versioned.post({
    access: 'public',
    path: _constants.RUN_SCRIPT_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.RunScriptActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteExecuteOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'runscript')));
}
function responseActionRequestHandler(endpointContext, command) {
  const logger = endpointContext.logFactory.get('responseActionsHandler');
  return async (context, req, res) => {
    logger.debug(() => `response action [${command}]:\n${(0, _stringify.stringify)(req.body)}`);
    const experimentalFeatures = endpointContext.experimentalFeatures;

    // Note:  because our API schemas are defined as module static variables (as opposed to a
    //        `getter` function), we need to include this additional validation here, since
    //        `agent_type` is included in the schema independent of the feature flag
    if (isThirdPartyFeatureDisabled(req.body.agent_type, experimentalFeatures)) {
      return (0, _error_handler.errorHandler)(logger, res, new _custom_http_request_error.CustomHttpRequestError(`[request body.agent_type]: feature is disabled`, 400));
    }
    const coreContext = await context.core;
    const user = coreContext.security.authc.getCurrentUser();
    const esClient = coreContext.elasticsearch.client.asInternalUser;
    const casesClient = await endpointContext.service.getCasesClient(req);
    const connectorActions = (await context.actions).getActionsClient();
    const responseActionsClient = (0, _services.getResponseActionsClient)(req.body.agent_type || 'endpoint', {
      esClient,
      casesClient,
      endpointService: endpointContext.service,
      username: (user === null || user === void 0 ? void 0 : user.username) || 'unknown',
      connectorActions: new _services.NormalizedExternalConnectorClient(connectorActions, logger)
    });
    try {
      var _ref;
      const action = await handleActionCreation(command, req.body, responseActionsClient);
      const {
        action: actionId,
        ...data
      } = action;
      const legacyResponseData = _constants2.responseActionsWithLegacyActionProperty.includes(command) ? {
        action: (_ref = actionId !== null && actionId !== void 0 ? actionId : data.id) !== null && _ref !== void 0 ? _ref : ''
      } : {};
      return res.ok({
        body: {
          ...legacyResponseData,
          data
        }
      });
    } catch (err) {
      return (0, _error_handler.errorHandler)(logger, res, err);
    }
  };
}
function isThirdPartyFeatureDisabled(agentType, experimentalFeatures) {
  return agentType === 'sentinel_one' && !experimentalFeatures.responseActionsSentinelOneV1Enabled || agentType === 'crowdstrike' && !experimentalFeatures.responseActionsCrowdstrikeManualHostIsolationEnabled || agentType === 'microsoft_defender_endpoint' && !experimentalFeatures.responseActionsMSDefenderEndpointEnabled;
}
async function handleActionCreation(command, body, responseActionsClient) {
  switch (command) {
    case 'isolate':
      return responseActionsClient.isolate(body);
    case 'unisolate':
      return responseActionsClient.release(body);
    case 'running-processes':
      return responseActionsClient.runningProcesses(body);
    case 'execute':
      return responseActionsClient.execute(body);
    case 'suspend-process':
      return responseActionsClient.suspendProcess(body);
    case 'kill-process':
      return responseActionsClient.killProcess(body);
    case 'get-file':
      return responseActionsClient.getFile(body);
    case 'upload':
      return responseActionsClient.upload(body);
    case 'scan':
      return responseActionsClient.scan(body);
    case 'runscript':
      return responseActionsClient.runscript(body);
    default:
      throw new _custom_http_request_error.CustomHttpRequestError(`No handler found for response action command: [${command}]`, 501);
  }
}
function redirectHandler(location) {
  return async (context, _req, res) => {
    const basePath = (await context.securitySolution).getServerBasePath();
    return res.custom({
      statusCode: 308,
      headers: {
        location: `${basePath}${location}`
      }
    });
  };
}