"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadAiConnectors = void 0;
var _inferenceEndpointUiCommon = require("@kbn/inference-endpoint-ui-common");
var _constants = require("@kbn/triggers-actions-ui-plugin/public/common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AllowedActionTypeIds = ['.bedrock', '.gen-ai', '.gemini', '.inference'];
const isAllowedConnector = connector => AllowedActionTypeIds.includes(connector.actionTypeId);
const isPreConfiguredInferenceConnector = connector => connector.actionTypeId === '.inference' && connector.isPreconfigured;
const isValidAiConnector = async (connector, deps) => {
  if (connector.isMissingSecrets) {
    return false;
  }
  if (!isAllowedConnector(connector)) {
    return false;
  }
  if (isPreConfiguredInferenceConnector(connector)) {
    var _connector$config;
    const inferenceId = (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.inferenceId;
    if (!inferenceId) {
      return false;
    }
    const exists = await (0, _inferenceEndpointUiCommon.isInferenceEndpointExists)(deps.http, inferenceId);
    if (!exists) {
      return false;
    }
  }
  return true;
};

/**
 * Loads all AI connectors that are valid, meaning that they don't miss secrets.
 * And in the case of the inference connector, that the inference endpoint exists.
 * @param http - The HTTP client to use for making requests.
 * @returns A promise that resolves to an array of valid AI connectors.
 */
const loadAiConnectors = async http => {
  const allConnectors = await (0, _constants.loadAllActions)({
    http
  });
  const aiConnectors = [];
  for (const connector of allConnectors) {
    const isValid = await isValidAiConnector(connector, {
      http
    });
    if (isValid) {
      aiConnectors.push(connector);
    }
  }
  return aiConnectors;
};
exports.loadAiConnectors = loadAiConnectors;