"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.compareRolesByName = void 0;
exports.transformElasticsearchRoleToRole = transformElasticsearchRoleToRole;
var _securityAuthorizationCore = require("@kbn/security-authorization-core");
var _securityAuthorizationCoreCommon = require("@kbn/security-authorization-core-common");
var _securityPluginTypesServer = require("@kbn/security-plugin-types-server");
var _constants = require("../../../common/constants");
var _errors = require("../../errors");
var _privilege_serializer = require("../privilege_serializer");
var _resource_serializer = require("../resource_serializer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isReservedPrivilege = app => app === _constants.RESERVED_PRIVILEGES_APPLICATION_WILDCARD;
const isWildcardPrivilege = app => app === _constants.PRIVILEGES_ALL_WILDCARD;
function transformElasticsearchRoleToRole({
  features,
  elasticsearchRole,
  name,
  application,
  logger,
  subFeaturePrivilegeIterator,
  replaceDeprecatedKibanaPrivileges
}) {
  const kibanaTransformResult = transformRoleApplicationsToKibanaPrivileges({
    features,
    roleApplications: elasticsearchRole.applications,
    application,
    logger,
    subFeaturePrivilegeIterator,
    replaceDeprecatedKibanaPrivileges
  });
  return {
    name,
    ...(elasticsearchRole.description && {
      description: elasticsearchRole.description
    }),
    metadata: elasticsearchRole.metadata,
    transient_metadata: elasticsearchRole.transient_metadata,
    elasticsearch: {
      cluster: elasticsearchRole.cluster,
      remote_cluster: elasticsearchRole.remote_cluster,
      indices: elasticsearchRole.indices,
      remote_indices: elasticsearchRole.remote_indices,
      run_as: elasticsearchRole.run_as
    },
    kibana: kibanaTransformResult.success ? kibanaTransformResult.value : [],
    _transform_error: [...(kibanaTransformResult.success ? [] : ['kibana'])],
    _unrecognized_applications: extractUnrecognizedApplicationNames(elasticsearchRole.applications, application)
  };
}
function transformRoleApplicationsToKibanaPrivileges({
  features,
  roleApplications,
  application,
  logger,
  subFeaturePrivilegeIterator,
  replaceDeprecatedKibanaPrivileges
}) {
  const roleKibanaApplications = roleApplications.filter(roleApplication => roleApplication.application === application || isReservedPrivilege(roleApplication.application) || isWildcardPrivilege(roleApplication.application));

  // if any application entry contains an empty resource, we throw an error
  if (roleKibanaApplications.some(entry => entry.resources.length === 0)) {
    throw new Error(`ES returned an application entry without resources, can't process this`);
  }

  // if there is an entry with the reserved privileges application wildcard
  // and there are privileges which aren't reserved, we won't transform these
  if (roleKibanaApplications.some(entry => (isReservedPrivilege(entry.application) || isWildcardPrivilege(entry.application)) && !entry.privileges.every(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedReservedPrivilege(privilege) || isWildcardPrivilege(privilege)))) {
    return {
      success: false
    };
  }

  // if there is a reserved privilege assigned to an application other than the reserved privileges application wildcard, we won't transform these.
  if (roleKibanaApplications.some(entry => !isReservedPrivilege(entry.application) && !isWildcardPrivilege(entry.application) && entry.privileges.some(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedReservedPrivilege(privilege)))) {
    return {
      success: false
    };
  }

  // if space privilege assigned globally, we can't transform these
  if (roleKibanaApplications.some(entry => entry.resources.includes(_securityPluginTypesServer.GLOBAL_RESOURCE) && entry.privileges.some(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedSpaceBasePrivilege(privilege)))) {
    return {
      success: false
    };
  }

  // if global base or reserved privilege assigned at a space, we can't transform these
  if (roleKibanaApplications.some(entry => !entry.resources.includes(_securityPluginTypesServer.GLOBAL_RESOURCE) && entry.privileges.some(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedGlobalBasePrivilege(privilege) || _privilege_serializer.PrivilegeSerializer.isSerializedReservedPrivilege(privilege)))) {
    return {
      success: false
    };
  }

  // if base privilege assigned with feature privileges, we won't transform these
  if (roleKibanaApplications.some(entry => entry.privileges.some(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedFeaturePrivilege(privilege)) && (entry.privileges.some(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedGlobalBasePrivilege(privilege)) || entry.privileges.some(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedSpaceBasePrivilege(privilege))))) {
    return {
      success: false
    };
  }

  // if any application entry contains the '*' resource in addition to another resource, we can't transform these
  if (roleKibanaApplications.some(entry => entry.resources.includes(_securityPluginTypesServer.GLOBAL_RESOURCE) && entry.resources.length > 1)) {
    return {
      success: false
    };
  }
  const allResources = roleKibanaApplications.filter(entry => !isReservedPrivilege(entry.application) && !isWildcardPrivilege(entry.application)).flatMap(entry => entry.resources);

  // if we have improperly formatted resource entries, we can't transform these
  if (allResources.some(resource => resource !== _securityPluginTypesServer.GLOBAL_RESOURCE && !_resource_serializer.ResourceSerializer.isSerializedSpaceResource(resource))) {
    return {
      success: false
    };
  }

  // if we have resources duplicated in entries, we won't transform these
  if (allResources.length !== getUniqueList(allResources).length) {
    return {
      success: false
    };
  }

  // if a feature privilege requires all spaces, but is assigned to other spaces, we won't transform these
  if (roleKibanaApplications.some(entry => !entry.resources.includes(_securityPluginTypesServer.GLOBAL_RESOURCE) && features.some(f => {
    var _f$privileges;
    return Object.entries((_f$privileges = f.privileges) !== null && _f$privileges !== void 0 ? _f$privileges : {}).some(([privName, featurePrivilege]) => featurePrivilege.requireAllSpaces && entry.privileges.includes(_privilege_serializer.PrivilegeSerializer.serializeFeaturePrivilege(f.id, privName)));
  }))) {
    return {
      success: false
    };
  }

  // if a feature privilege has been disabled we won't transform these
  if (roleKibanaApplications.some(entry => features.some(f => {
    var _f$privileges2;
    return Object.entries((_f$privileges2 = f.privileges) !== null && _f$privileges2 !== void 0 ? _f$privileges2 : {}).some(([privName, featurePrivilege]) => featurePrivilege.disabled && entry.privileges.includes(_privilege_serializer.PrivilegeSerializer.serializeFeaturePrivilege(f.id, privName)));
  }))) {
    return {
      success: false
    };
  }

  // try/catch block ensures graceful return on deserialize exceptions
  try {
    const transformResult = roleKibanaApplications.map(({
      resources,
      privileges
    }) => {
      const featurePrivileges = deserializeKibanaFeaturePrivileges({
        features,
        subFeaturePrivilegeIterator,
        serializedPrivileges: privileges,
        replaceDeprecatedKibanaPrivileges
      });

      // if we're dealing with a global entry, which we've ensured above is only possible if it's the only item in the array
      if (resources.length === 1 && resources[0] === _securityPluginTypesServer.GLOBAL_RESOURCE) {
        const reservedPrivileges = privileges.filter(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedReservedPrivilege(privilege));
        const basePrivileges = privileges.filter(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedGlobalBasePrivilege(privilege));
        return {
          ...(reservedPrivileges.length ? {
            _reserved: reservedPrivileges.map(privilege => _privilege_serializer.PrivilegeSerializer.deserializeReservedPrivilege(privilege))
          } : {}),
          base: basePrivileges.map(privilege => _privilege_serializer.PrivilegeSerializer.serializeGlobalBasePrivilege(privilege)),
          feature: featurePrivileges,
          spaces: ['*']
        };
      }
      const basePrivileges = privileges.filter(privilege => _privilege_serializer.PrivilegeSerializer.isSerializedSpaceBasePrivilege(privilege));
      return {
        base: basePrivileges.map(privilege => _privilege_serializer.PrivilegeSerializer.deserializeSpaceBasePrivilege(privilege)),
        feature: featurePrivileges,
        spaces: resources.map(resource => _resource_serializer.ResourceSerializer.deserializeSpaceResource(resource))
      };
    });
    return {
      success: true,
      value: transformResult
    };
  } catch (e) {
    logger.error(`Error transforming Elasticsearch role: ${(0, _errors.getDetailedErrorMessage)(e)}`);
    return {
      success: false
    };
  }
}
const extractUnrecognizedApplicationNames = (roleApplications, application) => {
  return getUniqueList(roleApplications.filter(roleApplication => roleApplication.application !== application && !isReservedPrivilege(roleApplication.application) && !isWildcardPrivilege(roleApplication.application)).map(roleApplication => roleApplication.application));
};
function getUniqueList(list) {
  return Array.from(new Set(list));
}
const compareRolesByName = (roleA, roleB) => {
  if (roleA.name < roleB.name) {
    return -1;
  }
  if (roleA.name > roleB.name) {
    return 1;
  }
  return 0;
};
exports.compareRolesByName = compareRolesByName;
function deserializeKibanaFeaturePrivileges({
  features,
  subFeaturePrivilegeIterator,
  serializedPrivileges,
  replaceDeprecatedKibanaPrivileges
}) {
  // Filter out deprecated features upfront to avoid going through ALL features within a loop.
  const deprecatedFeatures = replaceDeprecatedKibanaPrivileges ? features.filter(feature => feature.deprecated) : undefined;
  const result = {};
  for (const serializedPrivilege of serializedPrivileges) {
    if (!_privilege_serializer.PrivilegeSerializer.isSerializedFeaturePrivilege(serializedPrivilege)) {
      continue;
    }
    const {
      featureId,
      privilege: privilegeId
    } = _privilege_serializer.PrivilegeSerializer.deserializeFeaturePrivilege(serializedPrivilege);

    // If feature privileges are deprecated, replace them with non-deprecated feature privileges according to the
    // deprecation "mapping".
    const deprecatedFeature = deprecatedFeatures === null || deprecatedFeatures === void 0 ? void 0 : deprecatedFeatures.find(feature => feature.id === featureId);
    if (deprecatedFeature) {
      const privilege = getPrivilegeById(deprecatedFeature, privilegeId, subFeaturePrivilegeIterator);
      const replacedBy = privilege ? (0, _securityAuthorizationCore.getReplacedByForPrivilege)(privilegeId, privilege) : undefined;
      if (!replacedBy) {
        throw new Error(`A deprecated feature "${featureId}" is missing a replacement for the "${privilegeId}" privilege.`);
      }
      for (const reference of replacedBy) {
        result[reference.feature] = getUniqueList([...(result[reference.feature] || []), ...reference.privileges]);
      }
    } else {
      result[featureId] = getUniqueList([...(result[featureId] || []), privilegeId]);
    }
  }
  return result;
}
function getPrivilegeById(feature, privilegeId, subFeaturePrivilegeIterator) {
  for (const topLevelPrivilege of ['all', 'read']) {
    if (privilegeId === topLevelPrivilege || privilegeId === (0, _securityAuthorizationCoreCommon.getMinimalPrivilegeId)(topLevelPrivilege)) {
      var _feature$privileges;
      return (_feature$privileges = feature.privileges) === null || _feature$privileges === void 0 ? void 0 : _feature$privileges[topLevelPrivilege];
    }
  }

  // Don't perform license check as it should be done during feature registration (once we support
  // license checks for deprecated privileges).
  for (const subFeaturePrivilege of subFeaturePrivilegeIterator(feature, () => true)) {
    if (subFeaturePrivilege.id === privilegeId) {
      return subFeaturePrivilege;
    }
  }
}