"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSearchApiKeysRoutes = registerSearchApiKeysRoutes;
var _configSchema = require("@kbn/config-schema");
var _types = require("../../types");
var _get_key_by_id = require("../lib/get_key_by_id");
var _create_key = require("../lib/create_key");
var _privileges = require("../lib/privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const API_KEY_NAME = 'Unrestricted API Key';
function registerSearchApiKeysRoutes(router, logger) {
  router.post({
    path: _types.APIRoutes.API_KEY_VALIDITY,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    try {
      const core = await context.core;
      const client = core.elasticsearch.client.asCurrentUser;
      const apiKey = await (0, _get_key_by_id.getAPIKeyById)(request.body.id, client, logger);
      if (!apiKey) {
        return response.customError({
          body: {
            message: 'API key is not found.'
          },
          statusCode: 404
        });
      }
      return response.ok({
        body: {
          isValid: !apiKey.invalidated
        },
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (e) {
      logger.error(`Error fetching API Key`);
      logger.error(e);
      return response.customError({
        body: {
          message: e.message
        },
        statusCode: 500
      });
    }
  });
  router.post({
    path: _types.APIRoutes.API_KEYS,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {},
    options: {
      access: 'internal'
    }
  }, async (context, _request, response) => {
    try {
      const core = await context.core;
      const client = core.elasticsearch.client.asCurrentUser;
      const clusterHasApiKeys = await (0, _privileges.fetchClusterHasApiKeys)(client, logger);
      if (clusterHasApiKeys) {
        return response.customError({
          body: {
            message: 'Project already has API keys'
          },
          statusCode: 400
        });
      }
      const canCreateApiKeys = await (0, _privileges.fetchUserStartPrivileges)(client, logger);
      if (!canCreateApiKeys) {
        return response.customError({
          body: {
            message: 'User does not have required privileges'
          },
          statusCode: 403
        });
      }
      const apiKey = await (0, _create_key.createAPIKey)(API_KEY_NAME, client, logger);
      return response.ok({
        body: apiKey,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (e) {
      logger.error(`Error creating API Key`);
      logger.error(e);
      return response.customError({
        body: {
          message: e.message
        },
        statusCode: 500
      });
    }
  });
}