"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerFindRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _lib = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerFindRoute = (router, managementServicePromise) => {
  const referenceSchema = _configSchema.schema.object({
    type: _configSchema.schema.string(),
    id: _configSchema.schema.string()
  });
  const searchOperatorSchema = _configSchema.schema.oneOf([_configSchema.schema.literal('OR'), _configSchema.schema.literal('AND')], {
    defaultValue: 'OR'
  });
  const sortFieldSchema = _configSchema.schema.oneOf([_configSchema.schema.literal('created_at'), _configSchema.schema.literal('updated_at'), _configSchema.schema.literal('type')]);
  router.get({
    path: '/api/kibana/management/saved_objects/_find',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        perPage: _configSchema.schema.number({
          min: 0,
          defaultValue: 20
        }),
        page: _configSchema.schema.number({
          min: 0,
          defaultValue: 1
        }),
        type: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())]),
        search: _configSchema.schema.maybe(_configSchema.schema.string()),
        defaultSearchOperator: searchOperatorSchema,
        sortField: _configSchema.schema.maybe(sortFieldSchema),
        sortOrder: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('asc'), _configSchema.schema.literal('desc')])),
        hasReference: _configSchema.schema.maybe(_configSchema.schema.oneOf([referenceSchema, _configSchema.schema.arrayOf(referenceSchema)])),
        hasReferenceOperator: searchOperatorSchema
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const {
      query
    } = req;
    const managementService = await managementServicePromise;
    const {
      getClient,
      typeRegistry
    } = (await context.core).savedObjects;
    const searchTypes = Array.isArray(query.type) ? query.type : [query.type];
    const importAndExportableTypes = searchTypes.filter(type => typeRegistry.isImportableAndExportable(type));
    const includedHiddenTypes = importAndExportableTypes.filter(type => typeRegistry.isHidden(type));
    const client = getClient({
      includedHiddenTypes
    });
    const searchFields = new Set();
    importAndExportableTypes.forEach(type => {
      const searchField = managementService.getDefaultSearchField(type);
      if (searchField) {
        searchFields.add(searchField);
      }
    });
    const findResponse = await client.find({
      ...query,
      fields: undefined,
      searchFields: [...searchFields]
    });
    const savedObjects = findResponse.saved_objects.map(_lib.toSavedObjectWithMeta);
    const response = {
      saved_objects: savedObjects.map(so => {
        const obj = (0, _lib.injectMetaAttributes)(so, managementService);
        const result = {
          ...obj,
          attributes: {}
        };
        return result;
      }),
      total: findResponse.total,
      per_page: findResponse.per_page,
      page: findResponse.page
    };
    return res.ok({
      body: response
    });
  }));
};
exports.registerFindRoute = registerFindRoute;