"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResourceInstaller = void 0;
var _rxjs = require("rxjs");
var _server = require("@kbn/alerting-plugin/server");
var _assets = require("../../common/assets");
var _technical_component_template = require("../../common/assets/component_templates/technical_component_template");
var _ecs_component_template = require("../../common/assets/component_templates/ecs_component_template");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ResourceInstaller {
  constructor(options) {
    this.options = options;
  }

  // -----------------------------------------------------------------------------------------------
  // Common resources

  /**
   * Installs common, library-level resources shared between all indices:
   *   - default ILM policy
   *   - component template containing technical fields
   *   - component template containing all standard ECS fields
   */
  async installCommonResources() {
    await (0, _rxjs.firstValueFrom)(this.options.elasticsearchAndSOAvailability$.pipe((0, _rxjs.filter)(areESAndSOAvailable => areESAndSOAvailable)));
    const resourceDescription = 'common resources shared between all indices';
    const {
      logger,
      isWriteEnabled
    } = this.options;
    if (!isWriteEnabled) {
      logger.info(`Write is disabled; not installing ${resourceDescription}`);
      return;
    }
    await (0, _server.installWithTimeout)({
      description: resourceDescription,
      logger,
      pluginStop$: this.options.pluginStop$,
      installFn: async () => {
        const {
          getClusterClient,
          frameworkAlerts
        } = this.options;
        const clusterClient = await getClusterClient();
        try {
          // We can install them in parallel
          await Promise.all([
          // Install ILM policy and ECS component template only if framework alerts are not enabled
          // If framework alerts are enabled, the alerting framework will install these
          ...(frameworkAlerts.enabled() ? [] : [(0, _server.createOrUpdateIlmPolicy)({
            logger,
            esClient: clusterClient,
            name: _server.DEFAULT_ALERTS_ILM_POLICY_NAME,
            policy: _server.DEFAULT_ALERTS_ILM_POLICY,
            dataStreamAdapter: this.options.dataStreamAdapter
          }), (0, _server.createOrUpdateComponentTemplate)({
            logger,
            esClient: clusterClient,
            template: {
              name: _server.ECS_COMPONENT_TEMPLATE_NAME,
              body: _ecs_component_template.ecsComponentTemplate
            },
            totalFieldsLimit: _server.TOTAL_FIELDS_LIMIT
          })]), (0, _server.createOrUpdateComponentTemplate)({
            logger,
            esClient: clusterClient,
            template: {
              name: _assets.TECHNICAL_COMPONENT_TEMPLATE_NAME,
              body: _technical_component_template.technicalComponentTemplate
            },
            totalFieldsLimit: _server.TOTAL_FIELDS_LIMIT
          })]);
        } catch (err) {
          logger.error(`Error installing common resources in RuleRegistry ResourceInstaller - ${err.message}`);
          throw err;
        }
      }
    });
  }

  // -----------------------------------------------------------------------------------------------
  // Index-level resources

  /**
   * Installs index-level resources shared between all namespaces of this index:
   *   - custom ILM policy if it was provided
   *   - component templates
   */
  async installIndexLevelResources(indexInfo) {
    const resourceDescription = `resources for index ${indexInfo.baseName}`;
    const {
      logger,
      isWriteEnabled
    } = this.options;
    if (!isWriteEnabled) {
      logger.info(`Write is disabled; not installing ${resourceDescription}`);
      return;
    }
    await (0, _server.installWithTimeout)({
      description: resourceDescription,
      logger,
      pluginStop$: this.options.pluginStop$,
      installFn: async () => {
        const {
          frameworkAlerts,
          getClusterClient
        } = this.options;
        const {
          componentTemplates,
          ilmPolicy,
          additionalPrefix
        } = indexInfo.indexOptions;
        const clusterClient = await getClusterClient();

        // Rule registry allows for installation of custom ILM policy, which is only
        // used by security preview indices. We will continue to let rule registry
        // handle this installation.
        if (ilmPolicy != null) {
          await (0, _server.createOrUpdateIlmPolicy)({
            logger,
            esClient: clusterClient,
            name: indexInfo.getIlmPolicyName(),
            policy: ilmPolicy,
            dataStreamAdapter: this.options.dataStreamAdapter
          });
        }

        // Rule registry allows for installation of resources with an additional prefix,
        // which is only used by security preview indices. We will continue to let rule registry
        // handle installation of these resources.
        if (!frameworkAlerts.enabled() || additionalPrefix) {
          await Promise.all(componentTemplates.map(async ct => {
            var _ct$settings;
            await (0, _server.createOrUpdateComponentTemplate)({
              logger,
              esClient: clusterClient,
              template: {
                name: indexInfo.getComponentTemplateName(ct.name),
                body: {
                  template: {
                    settings: (_ct$settings = ct.settings) !== null && _ct$settings !== void 0 ? _ct$settings : {},
                    mappings: ct.mappings
                  },
                  _meta: ct._meta
                }
              },
              totalFieldsLimit: _server.TOTAL_FIELDS_LIMIT
            });
          }));
        }
      }
    });
  }

  // -----------------------------------------------------------------------------------------------
  // Namespace-level resources

  /**
   * Installs and updates resources tied to concrete namespace of an index:
   *   - namespaced index template
   *   - Index mappings for existing concrete indices
   *   - concrete index (write target) if it doesn't exist
   */
  async installAndUpdateNamespaceLevelResources(indexInfo, namespace) {
    const {
      logger,
      frameworkAlerts,
      getClusterClient
    } = this.options;
    const clusterClient = await getClusterClient();
    const alias = indexInfo.getPrimaryAlias(namespace);
    if (!indexInfo.indexOptions.additionalPrefix && frameworkAlerts.enabled()) {
      const {
        result: initialized,
        error
      } = await frameworkAlerts.getContextInitializationPromise(indexInfo.indexOptions.registrationContext, namespace);
      if (!initialized) {
        throw new Error(`There was an error in the framework installing namespace-level resources and creating concrete indices for ${alias} - ${error}`);
      } else {
        return;
      }
    }
    logger.info(`Installing namespace-level resources and creating concrete index for ${alias}`);
    const secondaryNamespacedAlias = indexInfo.getSecondaryAlias(namespace);
    const indexPatterns = {
      basePattern: indexInfo.basePattern,
      pattern: indexInfo.getPatternForBackingIndices(namespace),
      alias: indexInfo.getPrimaryAlias(namespace),
      name: indexInfo.getConcreteIndexInitialName(namespace),
      template: indexInfo.getIndexTemplateName(namespace),
      validPrefixes: _server.VALID_ALERT_INDEX_PREFIXES,
      ...(secondaryNamespacedAlias ? {
        secondaryAlias: secondaryNamespacedAlias
      } : {})
    };
    const technicalComponentNames = [_assets.TECHNICAL_COMPONENT_TEMPLATE_NAME];
    const ownComponentNames = indexInfo.indexOptions.componentTemplates.map(template => indexInfo.getComponentTemplateName(template.name));
    // Order matters:
    // - first go external component templates referenced by this index (e.g. the common full ECS template)
    // - then we include own component templates registered with this index
    // - finally, we include technical component templates to make sure the index gets all the
    //   mappings and settings required by all Kibana plugins using rule registry to work properly
    const componentTemplateRefs = [...indexInfo.indexOptions.componentTemplateRefs, ...ownComponentNames, ...technicalComponentNames];

    // Install / update the index template
    await (0, _server.createOrUpdateIndexTemplate)({
      logger: this.options.logger,
      esClient: clusterClient,
      template: (0, _server.getIndexTemplate)({
        componentTemplateRefs,
        ilmPolicyName: _server.DEFAULT_ALERTS_ILM_POLICY_NAME,
        indexPatterns,
        kibanaVersion: indexInfo.kibanaVersion,
        namespace,
        totalFieldsLimit: _server.TOTAL_FIELDS_LIMIT,
        dataStreamAdapter: this.options.dataStreamAdapter
      })
    });
    await (0, _server.createConcreteWriteIndex)({
      logger: this.options.logger,
      esClient: clusterClient,
      totalFieldsLimit: _server.TOTAL_FIELDS_LIMIT,
      indexPatterns,
      dataStreamAdapter: this.options.dataStreamAdapter
    });
  }
}
exports.ResourceInstaller = ResourceInstaller;