"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDeprecationsRoutes = void 0;
var _elasticsearch = require("@elastic/elasticsearch");
var _reportingCommon = require("@kbn/reporting-common");
var _reportingServer = require("@kbn/reporting-server");
var _lib = require("../../../lib");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAuthzWrapper = (reporting, logger) => handler => {
  return async (ctx, req, res) => {
    const {
      security
    } = reporting.getPluginSetupDeps();
    if (!(security !== null && security !== void 0 && security.license.isEnabled())) {
      return handler(ctx, req, res);
    }
    const {
      elasticsearch
    } = await ctx.core;
    try {
      const body = await elasticsearch.client.asCurrentUser.security.hasPrivileges({
        body: {
          index: [{
            privileges: ['manage'],
            // required to do anything with the reporting indices
            names: [_reportingServer.REPORTING_DATA_STREAM_WILDCARD_WITH_LEGACY],
            allow_restricted_indices: true
          }]
        }
      });
      if (!body.has_all_requested) {
        const responseString = JSON.stringify(body);
        logger.debug(`Current user does not have manage privilege for Reporting indices: ${responseString}`);
        return res.forbidden({
          body: `The current user requires "manage" privilege on the "${_reportingServer.REPORTING_DATA_STREAM_WILDCARD_WITH_LEGACY}" indices.`
        });
      }
    } catch (e) {
      logger.error(e);
      return res.customError({
        statusCode: e.statusCode,
        body: e.message
      });
    }
    return handler(ctx, req, res);
  };
};
const registerDeprecationsRoutes = (reporting, logger) => {
  const {
    router
  } = reporting.getPluginSetupDeps();
  const authzWrapper = getAuthzWrapper(reporting, logger);
  const getStatusPath = _reportingCommon.INTERNAL_ROUTES.MIGRATE.GET_ILM_POLICY_STATUS;
  router.get({
    path: getStatusPath,
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, authzWrapper(async ({
    core
  }, req, res) => {
    const counters = (0, _common.getCounters)(req.route.method, getStatusPath, reporting.getUsageCounter());
    const checkIlmMigrationStatus = async () => {
      const {
        client: scopedClient
      } = (await core).elasticsearch;
      const ilmPolicyManager = _lib.IlmPolicyManager.create({
        client: scopedClient.asInternalUser
      });
      return ilmPolicyManager.checkIlmMigrationStatus();
    };
    try {
      const response = {
        status: await checkIlmMigrationStatus()
      };
      counters.usageCounter();
      return res.ok({
        body: response
      });
    } catch (e) {
      var _e$statusCode;
      logger.error(e);
      const statusCode = (_e$statusCode = e === null || e === void 0 ? void 0 : e.statusCode) !== null && _e$statusCode !== void 0 ? _e$statusCode : 500;
      counters.errorCounter(statusCode);
      return res.customError({
        body: {
          message: e.message
        },
        statusCode
      });
    }
  }));
  const migrateApiPath = _reportingCommon.INTERNAL_ROUTES.MIGRATE.MIGRATE_ILM_POLICY;
  router.put({
    path: migrateApiPath,
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, authzWrapper(async ({
    core
  }, req, res) => {
    const counters = (0, _common.getCounters)(req.route.method, migrateApiPath, reporting.getUsageCounter());

    // First we ensure that the reporting ILM policy exists in the cluster
    try {
      const {
        client: {
          asCurrentUser
        }
      } = (await core).elasticsearch;
      const scopedIlmPolicyManager = _lib.IlmPolicyManager.create({
        client: asCurrentUser
      });

      // We don't want to overwrite an existing reporting policy because it may contain alterations made by users
      if (!(await scopedIlmPolicyManager.doesIlmPolicyExist())) {
        await scopedIlmPolicyManager.createIlmPolicy();
      }
    } catch (e) {
      var _e$statusCode2;
      return res.customError({
        statusCode: (_e$statusCode2 = e === null || e === void 0 ? void 0 : e.statusCode) !== null && _e$statusCode2 !== void 0 ? _e$statusCode2 : 500,
        body: {
          message: e.message
        }
      });
    }

    // Second we migrate all of the existing indices to be managed by the reporting ILM policy
    try {
      const {
        client: {
          asInternalUser
        }
      } = (await core).elasticsearch;
      const unscopedIlmPolicyManager = _lib.IlmPolicyManager.create({
        client: asInternalUser
      });
      const response = await unscopedIlmPolicyManager.migrateIndicesToIlmPolicy();
      counters.usageCounter();
      return res.ok({
        body: response
      });
    } catch (err) {
      logger.error(err);
      if (err instanceof _elasticsearch.errors.ResponseError) {
        var _err$statusCode;
        // If there were no reporting indices to update, that's OK because then there is nothing to migrate
        if (err.statusCode === 404) {
          counters.errorCounter(undefined, 404);
          return res.ok();
        }
        const statusCode = (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500;
        counters.errorCounter(undefined, statusCode);
        return res.customError({
          body: {
            message: err.message,
            name: err.name
          },
          statusCode
        });
      }
      throw err;
    }
  }));
};
exports.registerDeprecationsRoutes = registerDeprecationsRoutes;