"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.API_PRIVILEGES = void 0;
exports.registerFeatures = registerFeatures;
var _server = require("@kbn/core/server");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/features-plugin/common");
var _saved_objects = require("./saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const API_PRIVILEGES = exports.API_PRIVILEGES = {
  MANAGE_SCHEDULED_REPORTING: 'manage_scheduled_reports'
};
/**
 * If xpack.reporting.roles.enabled === true, register Reporting as a feature
 * that is controlled by user role names. Also, for Serverless register a
 * 'shell' Reporting Kibana feature.
 */
function registerFeatures({
  isServerless,
  features,
  deprecatedRoles,
  logger
}) {
  // Register a 'shell' feature specifically for Serverless. If granted, it will automatically provide access to
  // reporting capabilities in other features, such as Discover, Dashboards, and Visualizations. On its own, this
  // feature doesn't grant any additional privileges.
  if (isServerless) {
    features.registerKibanaFeature({
      id: 'reporting',
      name: _i18n.i18n.translate('xpack.reporting.features.reportingFeatureName', {
        defaultMessage: 'Reporting'
      }),
      category: _server.DEFAULT_APP_CATEGORIES.management,
      scope: [_common.KibanaFeatureScope.Spaces, _common.KibanaFeatureScope.Security],
      app: [],
      privileges: {
        all: {
          savedObject: {
            all: [],
            read: []
          },
          ui: []
        },
        // No read-only mode currently supported
        read: {
          disabled: true,
          savedObject: {
            all: [],
            read: []
          },
          ui: []
        }
      }
    });
  }
  if (deprecatedRoles !== false) {
    // refer to roles.allow configuration (deprecated path)
    const allowedRoles = ['superuser', ...(deprecatedRoles !== null && deprecatedRoles !== void 0 ? deprecatedRoles : [])];
    const privileges = allowedRoles.map(role => ({
      requiredClusterPrivileges: [],
      requiredRoles: [role],
      ui: []
    }));

    // self-register as an elasticsearch feature (deprecated)
    features.registerElasticsearchFeature({
      id: 'reporting',
      catalogue: ['reporting'],
      management: {
        insightsAndAlerting: ['reporting']
      },
      privileges
    });
  } else {
    logger.debug(`Reporting roles configuration is disabled. Please assign access to Reporting use Kibana feature controls for applications.`);
    // trigger application to register Reporting as a subfeature
    features.enableReportingUiCapabilities();
  }
  features.registerKibanaFeature({
    id: 'manageReporting',
    name: _i18n.i18n.translate('xpack.reporting.features.manageScheduledReportsFeatureName', {
      defaultMessage: 'Manage Scheduled Reports'
    }),
    description: _i18n.i18n.translate('xpack.reporting.features.manageScheduledReportsFeatureDescription', {
      defaultMessage: 'View and manage scheduled reports for all users in this space.'
    }),
    category: _server.DEFAULT_APP_CATEGORIES.management,
    scope: [_common.KibanaFeatureScope.Spaces, _common.KibanaFeatureScope.Security],
    app: [],
    privileges: {
      all: {
        api: [API_PRIVILEGES.MANAGE_SCHEDULED_REPORTING],
        savedObject: {
          all: [_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE],
          read: []
        },
        ui: ['show']
      },
      // No read-only mode currently supported
      read: {
        disabled: true,
        savedObject: {
          all: [],
          read: []
        },
        ui: []
      }
    }
  });
}