"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAnnotationsClient = createAnnotationsClient;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _format_annotations = require("./format_annotations");
var _permissions = require("./permissions");
var _annotation_mappings = require("./mappings/annotation_mappings");
var _annotations = require("../../../common/annotations");
var _create_or_update_index = require("../../utils/create_or_update_index");
var _unwrap_es_response = require("../../../common/utils/unwrap_es_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createAnnotationsClient(params) {
  const {
    index,
    esClient,
    logger,
    license
  } = params;
  const readIndex = index === _annotations.DEFAULT_ANNOTATION_INDEX ? index : `${index},${_annotations.DEFAULT_ANNOTATION_INDEX}`;
  const initIndex = () => (0, _create_or_update_index.createOrUpdateIndex)({
    index,
    client: esClient,
    logger,
    mappings: _annotation_mappings.ANNOTATION_MAPPINGS
  });
  function ensureGoldLicense(fn) {
    return (...args) => {
      if (!(license !== null && license !== void 0 && license.hasAtLeast('gold'))) {
        throw _boom.default.forbidden('Annotations require at least a gold license or a trial license.');
      }
      return fn(...args);
    };
  }
  const updateMappings = async () => {
    var _mappings$properties;
    // get index mapping
    const currentMappings = await esClient.indices.getMapping({
      index
    });
    const mappings = currentMappings === null || currentMappings === void 0 ? void 0 : currentMappings[index].mappings;
    if (mappings !== null && mappings !== void 0 && (_mappings$properties = mappings.properties) !== null && _mappings$properties !== void 0 && _mappings$properties.slo) {
      return;
    }

    // update index mapping
    await initIndex();
  };
  const validateAnnotation = annotation => {
    // make sure to check one of message of annotation.title is present
    if (!annotation.message && !annotation.annotation.title) {
      throw _boom.default.badRequest('Annotation must have a message or a annotation.title');
    }
  };
  return {
    index,
    create: ensureGoldLicense(async createParams => {
      validateAnnotation(createParams);
      const indexExists = await (0, _unwrap_es_response.unwrapEsResponse)(esClient.indices.exists({
        index
      }, {
        meta: true
      }));
      if (!indexExists) {
        await initIndex();
      } else {
        await updateMappings();
      }
      const annotation = {
        ...createParams,
        event: {
          ...createParams.event,
          created: new Date().toISOString()
        }
      };
      if (!annotation.annotation.title) {
        // TODO: handle this when we integrate with the APM UI
        annotation.annotation.title = annotation.message;
      }
      const body = await (0, _unwrap_es_response.unwrapEsResponse)(esClient.index({
        index,
        body: annotation,
        refresh: 'wait_for'
      }, {
        meta: true
      }));
      const document = (await esClient.get({
        index,
        id: body._id
      }, {
        meta: true
      })).body;
      return {
        _id: document._id,
        _index: document._index,
        _source: (0, _format_annotations.formatAnnotation)(document._source)
      };
    }),
    update: ensureGoldLicense(async updateParams => {
      validateAnnotation(updateParams);
      await updateMappings();
      const {
        id,
        ...rest
      } = updateParams;
      const annotation = {
        ...rest,
        event: {
          ...rest.event,
          updated: new Date().toISOString()
        }
      };
      if (!annotation.annotation.title) {
        // TODO: handle this when we integrate with the APM UI
        annotation.annotation.title = annotation.message;
      }
      const body = await (0, _unwrap_es_response.unwrapEsResponse)(esClient.index({
        index,
        id,
        body: annotation,
        refresh: 'wait_for'
      }, {
        meta: true
      }));
      return (await esClient.get({
        index,
        id: body._id
      }, {
        meta: true
      })).body;
    }),
    getById: ensureGoldLicense(async getByIdParams => {
      var _response$hits$hits;
      const {
        id
      } = getByIdParams;
      const response = await esClient.search({
        index: readIndex,
        ignore_unavailable: true,
        query: {
          bool: {
            filter: [{
              term: {
                _id: id
              }
            }]
          }
        }
      });
      const document = (_response$hits$hits = response.hits.hits) === null || _response$hits$hits === void 0 ? void 0 : _response$hits$hits[0];
      return {
        ...document,
        _source: (0, _format_annotations.formatAnnotation)(document._source)
      };
    }),
    find: ensureGoldLicense(async findParams => {
      var _result$hits$total;
      const {
        start,
        end,
        sloId,
        sloInstanceId,
        serviceName,
        filter,
        size
      } = findParams !== null && findParams !== void 0 ? findParams : {};
      const filterJSON = filter ? JSON.parse(filter) : {};
      const termsFilter = Object.keys(filterJSON).map(filterKey => ({
        term: {
          [filterKey]: filterJSON[filterKey]
        }
      }));
      const shouldClauses = [];
      if (sloId || sloInstanceId) {
        const query = {
          bool: {
            should: [{
              term: {
                'slo.id': '*'
              }
            }, {
              bool: {
                filter: [...(sloId ? [{
                  match_phrase: {
                    'slo.id': sloId
                  }
                }] : []), ...(sloInstanceId ? [{
                  match_phrase: {
                    'slo.instanceId': sloInstanceId
                  }
                }] : [])]
              }
            }]
          }
        };
        shouldClauses.push(query);
      }
      const result = await esClient.search({
        index: readIndex,
        size: size !== null && size !== void 0 ? size : 10000,
        ignore_unavailable: true,
        query: {
          bool: {
            filter: [{
              range: {
                '@timestamp': {
                  gte: start !== null && start !== void 0 ? start : 'now-30d',
                  lte: end !== null && end !== void 0 ? end : 'now'
                }
              }
            }, ...(Object.keys(filterJSON).length !== 0 ? termsFilter : [{
              bool: {
                should: [...(serviceName ? [{
                  term: {
                    'service.name': serviceName
                  }
                }] : []), ...shouldClauses]
              }
            }])]
          }
        }
      });
      const items = result.hits.hits.map(hit => ({
        ...(0, _format_annotations.formatAnnotation)(hit._source),
        id: hit._id
      }));
      return {
        items,
        total: (_result$hits$total = result.hits.total) === null || _result$hits$total === void 0 ? void 0 : _result$hits$total.value
      };
    }),
    delete: ensureGoldLicense(async deleteParams => {
      const {
        id
      } = deleteParams;
      return await esClient.deleteByQuery({
        index: readIndex,
        ignore_unavailable: true,
        body: {
          query: {
            term: {
              _id: id
            }
          }
        },
        refresh: true
      });
    }),
    permissions: async () => {
      var _license$hasAtLeast;
      const permissions = await (0, _permissions.checkAnnotationsPermissions)({
        index,
        esClient
      });
      const hasGoldLicense = (_license$hasAtLeast = license === null || license === void 0 ? void 0 : license.hasAtLeast('gold')) !== null && _license$hasAtLeast !== void 0 ? _license$hasAtLeast : false;
      return {
        index,
        hasGoldLicense,
        ...permissions.index[index]
      };
    }
  };
}