"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.recallAndScore = recallAndScore;
var _score_suggestions = require("./score_suggestions");
var _recall_ranking = require("../../../analytics/recall_ranking");
var _query_rewrite = require("./query_rewrite");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function recallAndScore({
  recall,
  chat,
  analytics,
  screenDescription,
  messages,
  logger,
  signal
}) {
  const rewrittenUserPrompt = await (0, _query_rewrite.queryRewrite)({
    screenDescription,
    chat,
    messages,
    logger,
    signal
  });
  const queries = [{
    text: rewrittenUserPrompt,
    boost: 1
  }];
  const suggestions = (await recall({
    queries
  })).map(({
    id,
    text,
    esScore
  }) => ({
    id,
    text,
    esScore
  }));
  if (!suggestions.length) {
    logger.debug('No suggestions found during recall');
    return {
      relevantDocuments: [],
      llmScores: [],
      suggestions: []
    };
  }
  logger.debug(`Found ${suggestions.length} suggestions during recall`);
  try {
    const {
      llmScores,
      relevantDocuments
    } = await (0, _score_suggestions.scoreSuggestions)({
      suggestions,
      logger,
      messages,
      screenDescription,
      signal,
      chat
    });
    logger.debug(`Found ${relevantDocuments.length} relevant documents out of ${suggestions.length} suggestions`);
    analytics.reportEvent(_recall_ranking.recallRankingEventType, {
      scoredDocuments: suggestions.map(suggestion => {
        var _suggestion$esScore;
        const llmScore = llmScores.find(score => score.id === suggestion.id);
        return {
          esScore: (_suggestion$esScore = suggestion.esScore) !== null && _suggestion$esScore !== void 0 ? _suggestion$esScore : -1,
          llmScore: llmScore ? llmScore.llmScore : -1
        };
      })
    });
    return {
      llmScores,
      relevantDocuments,
      suggestions
    };
  } catch (error) {
    logger.error(`Error scoring documents: ${error.message}`, {
      error
    });
    return {
      suggestions: suggestions.slice(0, 5)
    };
  }
}