"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dataFeedRoutes = dataFeedRoutes;
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _datafeeds_schema = require("./schemas/datafeeds_schema");
var _request_authorization = require("../lib/request_authorization");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Routes for datafeed service
 */
function dataFeedRoutes({
  router,
  routeGuard
}) {
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetDatafeeds']
      }
    },
    summary: 'Gets all datafeeds',
    description: 'Retrieves configuration information for datafeeds.'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    response
  }) => {
    try {
      const body = await mlClient.getDatafeeds();
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetDatafeeds']
      }
    },
    summary: 'Get datafeed for given datafeed id',
    description: 'Retrieves configuration information for a datafeed.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const datafeedId = request.params.datafeedId;
      const body = await mlClient.getDatafeeds({
        datafeed_id: datafeedId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/_stats`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetDatafeeds']
      }
    },
    summary: 'Gets stats for all datafeeds',
    description: 'Retrieves usage information for datafeeds.'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    response
  }) => {
    try {
      const body = await mlClient.getDatafeedStats();
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}/_stats`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetDatafeeds']
      }
    },
    summary: 'Get datafeed stats for given datafeed id',
    description: 'Retrieves usage information for a datafeed.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const datafeedId = request.params.datafeedId;
      const body = await mlClient.getDatafeedStats({
        datafeed_id: datafeedId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.put({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateDatafeed']
      }
    },
    summary: 'Creates a datafeed',
    description: 'Instantiates a datafeed.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema,
        body: _datafeeds_schema.datafeedConfigSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const datafeedId = request.params.datafeedId;
      const body = await mlClient.putDatafeed({
        datafeed_id: datafeedId,
        // @ts-expect-error type mismatch for `time_span` (string | number versus estypes.Duration)
        body: request.body
      }, (0, _request_authorization.getAuthorizationHeader)(request));
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}/_update`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canUpdateDatafeed']
      }
    },
    summary: 'Updates a datafeed',
    description: 'Updates certain properties of a datafeed.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema,
        body: _datafeeds_schema.datafeedConfigSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const datafeedId = request.params.datafeedId;
      const body = await mlClient.updateDatafeed({
        datafeed_id: datafeedId,
        // @ts-expect-error type mismatch for `time_span` (string | number versus estypes.Duration)
        body: request.body
      }, (0, _request_authorization.getAuthorizationHeader)(request));
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.delete({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canDeleteDatafeed']
      }
    },
    summary: 'Deletes a datafeed',
    description: 'Deletes an existing datafeed.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema,
        query: _datafeeds_schema.deleteDatafeedQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const options = {
        datafeed_id: request.params.datafeedId
      };
      const force = request.query.force;
      if (force !== undefined) {
        options.force = force;
      }
      const body = await mlClient.deleteDatafeed(options);
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}/_start`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canStartStopDatafeed']
      }
    },
    summary: 'Starts a datafeed',
    description: 'Starts one or more datafeeds'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema,
        body: _datafeeds_schema.startDatafeedSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const datafeedId = request.params.datafeedId;
      const {
        start,
        end
      } = request.body;
      const body = await mlClient.startDatafeed({
        datafeed_id: datafeedId,
        body: {
          start: start !== undefined ? String(start) : undefined,
          end: end !== undefined ? String(end) : undefined
        }
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}/_stop`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canStartStopDatafeed']
      }
    },
    summary: 'Stops a datafeed',
    description: 'Stops one or more datafeeds'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const datafeedId = request.params.datafeedId;
      const body = await mlClient.stopDatafeed({
        datafeed_id: datafeedId
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/{datafeedId}/_preview`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canPreviewDatafeed']
      }
    },
    summary: 'Previews a datafeed',
    description: 'Previews a datafeed'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _datafeeds_schema.datafeedIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const datafeedId = request.params.datafeedId;
      const body = await mlClient.previewDatafeed({
        datafeed_id: datafeedId
      }, {
        ...(0, _request_authorization.getAuthorizationHeader)(request),
        maxRetries: 0
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}