"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.modelsProvider = exports.ModelsProvider = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _mlDataFrameAnalyticsUtils = require("@kbn/ml-data-frame-analytics-utils");
var _lodash = require("lodash");
var _mlTrainedModelsUtils = require("@kbn/ml-trained-models-utils");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _trained_models = require("../../../common/constants/trained_models");
var _trained_models2 = require("../../../common/types/trained_models");
var _trained_models3 = require("../../routes/trained_models");
var _log = require("../../lib/log");
var _get_model_state = require("./get_model_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const modelsProvider = (client, mlClient, cloud, enabledFeatures) => new ModelsProvider(client, mlClient, cloud, enabledFeatures);
exports.modelsProvider = modelsProvider;
class ModelsProvider {
  constructor(_client, _mlClient, _cloud, _enabledFeatures) {
    (0, _defineProperty2.default)(this, "_transforms", void 0);
    this._client = _client;
    this._mlClient = _mlClient;
    this._cloud = _cloud;
    this._enabledFeatures = _enabledFeatures;
  }
  async initTransformData() {
    if (!this._transforms) {
      try {
        const body = await this._client.asCurrentUser.transform.getTransform({
          size: 1000
        });
        this._transforms = body.transforms;
        return body.transforms;
      } catch (e) {
        var _e$meta;
        if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) !== 403) {
          // eslint-disable-next-line no-console
          console.error(e);
        }
      }
    }
  }
  async getIndexData(index) {
    try {
      const indexData = await this._client.asInternalUser.indices.get({
        index
      });
      return indexData;
    } catch (e) {
      var _e$meta2;
      // Possible that the user doesn't have permissions to view
      // If so, gracefully exit
      if (((_e$meta2 = e.meta) === null || _e$meta2 === void 0 ? void 0 : _e$meta2.statusCode) !== 403) {
        // eslint-disable-next-line no-console
        console.error(e);
      }
      return {
        [index]: null
      };
    }
  }
  getNodeId(elementOriginalId, nodeType) {
    return `${elementOriginalId}-${nodeType}`;
  }

  /**
   * Assigns inference endpoints to trained models
   * @param trainedModels
   * @param asInternal
   */
  async assignInferenceEndpoints(trainedModels, asInternal = false) {
    const esClient = asInternal ? this._client.asInternalUser : this._client.asCurrentUser;
    try {
      // Check if model is used by an inference service
      const {
        endpoints
      } = await esClient.inference.get({
        inference_id: '_all'
      });
      const inferenceAPIMap = (0, _lodash.groupBy)(endpoints, endpoint => (endpoint.service === 'elser' || endpoint.service === 'elasticsearch') && endpoint.service_settings.model_id);
      for (const model of trainedModels) {
        const inferenceApis = inferenceAPIMap[model.model_id];
        model.hasInferenceServices = !!inferenceApis;
        if (model.hasInferenceServices && !asInternal) {
          model.inference_apis = inferenceApis;
        }
      }
    } catch (e) {
      if (!asInternal && e.statusCode === 403) {
        // retry with internal user to get an indicator if models has associated inference services, without mentioning the names
        await this.assignInferenceEndpoints(trainedModels, true);
      } else {
        _log.mlLog.error(e);
      }
    }
  }

  /**
   * Assigns trained model stats to trained models
   * @param trainedModels
   */
  async assignModelStats(trainedModels) {
    const {
      trained_model_stats: modelsStatsResponse
    } = await this._mlClient.getTrainedModelsStats({
      size: _trained_models.DEFAULT_TRAINED_MODELS_PAGE_SIZE
    });
    const groupByModelId = (0, _lodash.groupBy)(modelsStatsResponse, 'model_id');
    return trainedModels.map(model => {
      const modelStats = groupByModelId[model.model_id];
      const completeModelItem = {
        ...model,
        // @ts-ignore FIXME: fix modelStats type
        stats: {
          ...modelStats[0],
          ...((0, _trained_models2.isNLPModelItem)(model) ? {
            deployment_stats: modelStats.map(d => d.deployment_stats).filter(_mlIsDefined.isDefined)
          } : {})
        }
      };
      if ((0, _trained_models2.isNLPModelItem)(completeModelItem)) {
        // Extract deployment ids from deployment stats
        completeModelItem.deployment_ids = modelStats.map(v => {
          var _v$deployment_stats;
          return (_v$deployment_stats = v.deployment_stats) === null || _v$deployment_stats === void 0 ? void 0 : _v$deployment_stats.deployment_id;
        }).filter(_mlIsDefined.isDefined);
        completeModelItem.state = (0, _get_model_state.getModelDeploymentState)(completeModelItem);
        completeModelItem.stateDescription = completeModelItem.stats.deployment_stats.reduce((acc, c) => {
          var _c$reason;
          if (acc) return acc;
          return (_c$reason = c.reason) !== null && _c$reason !== void 0 ? _c$reason : '';
        }, '');
      }
      return completeModelItem;
    });
  }

  /**
   * Merges the list of models with the list of models available for download.
   */
  async includeModelDownloads(resultItems) {
    const idMap = new Map(resultItems.map(model => [model.model_id, model]));
    const [rawModels, forDownload] = await Promise.all([this._client.asCurrentUser.ml.getTrainedModels({
      size: 1000
    }), this.getModelDownloads()]);
    const allExistingModelIds = new Set(rawModels.trained_model_configs.map(m => m.model_id));
    const notDownloaded = forDownload.filter(({
      model_id: modelId,
      hidden,
      recommended,
      supported,
      disclaimer,
      techPreview
    }) => {
      if (idMap.has(modelId)) {
        const model = idMap.get(modelId);
        if (recommended) {
          model.recommended = true;
        }
        if (techPreview) {
          model.techPreview = true;
        }
        model.supported = supported;
        model.disclaimer = disclaimer;
      }
      return !idMap.has(modelId) && !hidden;
    }).map(modelDefinition => {
      var _modelDefinition$type;
      // Check if this downloadable model already exists in the system, but not in current space
      const isDownloadedWithinDifferentSpace = allExistingModelIds.has(modelDefinition.model_id);
      return {
        model_id: modelDefinition.model_id,
        type: modelDefinition.type,
        tags: (_modelDefinition$type = modelDefinition.type) !== null && _modelDefinition$type !== void 0 && _modelDefinition$type.includes(_mlTrainedModelsUtils.ELASTIC_MODEL_TAG) ? [_mlTrainedModelsUtils.ELASTIC_MODEL_TAG] : [],
        putModelConfig: modelDefinition.config,
        description: modelDefinition.description,
        state: isDownloadedWithinDifferentSpace ? _mlTrainedModelsUtils.MODEL_STATE.DOWNLOADED_IN_DIFFERENT_SPACE : _mlTrainedModelsUtils.MODEL_STATE.NOT_DOWNLOADED,
        recommended: !!modelDefinition.recommended,
        modelName: modelDefinition.modelName,
        os: modelDefinition.os,
        arch: modelDefinition.arch,
        softwareLicense: modelDefinition.license,
        licenseUrl: modelDefinition.licenseUrl,
        supported: modelDefinition.supported,
        disclaimer: modelDefinition.disclaimer
      };
    });

    // show model downloads first
    return [...notDownloaded, ...resultItems];
  }

  /**
   * Assigns pipelines to trained models
   */
  async assignPipelines(trainedModels) {
    // For each model create a dict with model aliases and deployment ids for faster lookup
    const modelToAliasesAndDeployments = Object.fromEntries(trainedModels.map(model => {
      var _model$metadata$model, _model$metadata;
      return [model.model_id, new Set([model.model_id, ...((_model$metadata$model = (_model$metadata = model.metadata) === null || _model$metadata === void 0 ? void 0 : _model$metadata.model_aliases) !== null && _model$metadata$model !== void 0 ? _model$metadata$model : []), ...((0, _trained_models2.isNLPModelItem)(model) ? model.deployment_ids : [])])];
    }));

    // Set of unique model ids, aliases, and deployment ids.
    const modelIdsAndAliases = Object.values(modelToAliasesAndDeployments).flatMap(s => Array.from(s));
    try {
      // Get all pipelines first in one call:
      const modelPipelinesMap = await this.getModelsPipelines(modelIdsAndAliases);
      trainedModels.forEach(model => {
        const modelAliasesAndDeployments = modelToAliasesAndDeployments[model.model_id];
        // Check model pipelines map for any pipelines associated with the model
        for (const [modelEntityId, pipelines] of modelPipelinesMap) {
          if (modelAliasesAndDeployments.has(modelEntityId)) {
            // Merge pipeline definitions into the model
            model.pipelines = model.pipelines ? Object.assign(model.pipelines, pipelines) : pipelines;
          }
        }
      });
    } catch (e) {
      // the user might not have required permissions to fetch pipelines
      // log the error to the debug log as this might be a common situation and
      // we don't need to fill kibana's log with these messages.
      _log.mlLog.debug(e);
    }
  }

  /**
   * Assigns indices to trained models
   */
  async assignModelIndices(trainedModels) {
    // Get a list of all uniquer pipeline ids to retrieve mapping with indices
    const pipelineIds = new Set(trainedModels.filter(model => (0, _mlIsDefined.isDefined)(model.pipelines)).flatMap(model => Object.keys(model.pipelines)));
    const pipelineToIndicesMap = await this.getPipelineToIndicesMap(pipelineIds);
    trainedModels.forEach(model => {
      if (!(0, _lodash.isEmpty)(model.pipelines)) {
        model.indices = Object.entries(pipelineToIndicesMap).filter(([pipelineId]) => {
          var _model$pipelines;
          return !(0, _lodash.isEmpty)((_model$pipelines = model.pipelines) === null || _model$pipelines === void 0 ? void 0 : _model$pipelines[pipelineId]);
        }).flatMap(([_, indices]) => indices);
      }
    });
  }

  /**
   * Assign a check for each DFA model if origin job exists
   */
  async assignDFAJobCheck(trainedModels) {
    try {
      const dfaJobIds = trainedModels.map(model => {
        var _model$metadata2, _model$metadata2$anal;
        const id = (_model$metadata2 = model.metadata) === null || _model$metadata2 === void 0 ? void 0 : (_model$metadata2$anal = _model$metadata2.analytics_config) === null || _model$metadata2$anal === void 0 ? void 0 : _model$metadata2$anal.id;
        if (id) {
          return `${id}*`;
        }
      }).filter(_mlIsDefined.isDefined);
      if (dfaJobIds.length > 0) {
        const {
          data_frame_analytics: jobs
        } = await this._mlClient.getDataFrameAnalytics({
          id: dfaJobIds.join(','),
          allow_no_match: true
        });
        trainedModels.forEach(model => {
          var _model$metadata3, _model$metadata3$anal;
          const dfaId = model === null || model === void 0 ? void 0 : (_model$metadata3 = model.metadata) === null || _model$metadata3 === void 0 ? void 0 : (_model$metadata3$anal = _model$metadata3.analytics_config) === null || _model$metadata3$anal === void 0 ? void 0 : _model$metadata3$anal.id;
          if (dfaId !== undefined) {
            // if this is a dfa model, set origin_job_exists
            model.origin_job_exists = jobs.find(job => job.id === dfaId) !== undefined;
          }
        });
      }
    } catch (e) {
      return;
    }
  }

  /**
   * Returns a complete list of entities for the Trained Models UI
   */
  async getTrainedModelList() {
    const resp = await this._mlClient.getTrainedModels({
      size: 1000
    });
    let resultItems = [];

    // Filter models based on enabled features
    const filteredModels = (0, _trained_models3.filterForEnabledFeatureModels)(resp.trained_model_configs, this._enabledFeatures);
    const formattedModels = filteredModels.map(model => {
      return {
        ...model,
        // Extract model types
        type: [model.model_type, ...((0, _trained_models2.isBuiltInModel)(model) ? [_mlTrainedModelsUtils.BUILT_IN_MODEL_TYPE] : []), ...((0, _trained_models2.isElasticModel)(model) ? [_mlTrainedModelsUtils.ELASTIC_MODEL_TYPE] : []), ...(typeof model.inference_config === 'object' ? Object.keys(model.inference_config) : [])].filter(_mlIsDefined.isDefined)
      };
    });

    // Update inference endpoints info
    await this.assignInferenceEndpoints(formattedModels);

    // Assign model stats
    resultItems = await this.assignModelStats(formattedModels);
    if (this._enabledFeatures.nlp) {
      resultItems = await this.includeModelDownloads(resultItems);
    }
    const existingModels = resultItems.filter(_trained_models2.isExistingModel);

    // Assign pipelines to existing models
    await this.assignPipelines(existingModels);

    // Assign indices
    await this.assignModelIndices(existingModels);
    await this.assignDFAJobCheck(resultItems.filter(_trained_models2.isDFAModelItem));
    return resultItems;
  }

  /**
   * Simulates the effect of the pipeline on given document.
   *
   */
  async simulatePipeline(docs, pipelineConfig) {
    const simulateRequest = {
      docs,
      pipeline: pipelineConfig
    };
    let result = {};
    try {
      result = await this._client.asCurrentUser.ingest.simulate(simulateRequest);
    } catch (error) {
      if (error.statusCode === 404) {
        // ES returns 404 when there are no pipelines
        // Instead, we should return an empty response and a 200
        return result;
      }
      throw error;
    }
    return result;
  }

  /**
   * Creates the pipeline
   *
   */
  async createInferencePipeline(pipelineConfig, pipelineName) {
    return await this._client.asCurrentUser.ingest.putPipeline({
      id: pipelineName,
      ...pipelineConfig
    });
  }

  /**
   * Retrieves existing pipelines.
   *
   */
  async getPipelines() {
    const result = await this._client.asCurrentUser.ingest.getPipeline({
      summary: true
    }, {
      ignore: [404]
    });
    return Object.keys(result);
  }

  /**
   * Retrieves the map of model ids and aliases with associated pipelines,
   * where key is a model, alias or deployment id, and value is a map of pipeline ids and pipeline definitions.
   * @param modelIds - Array of models ids and model aliases.
   */
  async getModelsPipelines(modelIds) {
    const modelIdsMap = new Map(modelIds.map(id => [id, {}]));
    try {
      const body = await this._client.asCurrentUser.ingest.getPipeline();
      for (const [pipelineName, pipelineDefinition] of Object.entries(body)) {
        const {
          processors
        } = pipelineDefinition;
        for (const processor of processors) {
          var _processor$inference;
          const id = (_processor$inference = processor.inference) === null || _processor$inference === void 0 ? void 0 : _processor$inference.model_id;
          if (modelIdsMap.has(id)) {
            const obj = modelIdsMap.get(id);
            if (obj === null) {
              modelIdsMap.set(id, {
                [pipelineName]: pipelineDefinition
              });
            } else {
              obj[pipelineName] = pipelineDefinition;
            }
          }
        }
      }
    } catch (error) {
      if (error.statusCode === 404) {
        // ES returns 404 when there are no pipelines
        // Instead, we should return the modelIdsMap and a 200
        return modelIdsMap;
      }
      throw error;
    }
    return modelIdsMap;
  }

  /**
   * Match pipelines to indices based on the default_pipeline setting in the index settings.
   */
  async getPipelineToIndicesMap(pipelineIds) {
    const pipelineIdsToDestinationIndices = {};
    let indicesPermissions;
    let indicesSettings;
    try {
      indicesSettings = await this._client.asInternalUser.indices.getSettings();
      const hasPrivilegesResponse = await this._client.asCurrentUser.security.hasPrivileges({
        index: [{
          names: Object.keys(indicesSettings),
          privileges: ['read']
        }]
      });
      indicesPermissions = hasPrivilegesResponse.index;
    } catch (e) {
      var _e$meta3;
      // Possible that the user doesn't have permissions to view
      if (((_e$meta3 = e.meta) === null || _e$meta3 === void 0 ? void 0 : _e$meta3.statusCode) !== 403) {
        _log.mlLog.error(e);
      }
      return pipelineIdsToDestinationIndices;
    }

    // From list of model pipelines, find all indices that have pipeline set as index.default_pipeline
    for (const [indexName, {
      settings
    }] of Object.entries(indicesSettings)) {
      var _settings$index, _indicesPermissions$i;
      const defaultPipeline = settings === null || settings === void 0 ? void 0 : (_settings$index = settings.index) === null || _settings$index === void 0 ? void 0 : _settings$index.default_pipeline;
      if (defaultPipeline && pipelineIds.has(defaultPipeline) && ((_indicesPermissions$i = indicesPermissions[indexName]) === null || _indicesPermissions$i === void 0 ? void 0 : _indicesPermissions$i.read) === true) {
        if (Array.isArray(pipelineIdsToDestinationIndices[defaultPipeline])) {
          pipelineIdsToDestinationIndices[defaultPipeline].push(indexName);
        } else {
          pipelineIdsToDestinationIndices[defaultPipeline] = [indexName];
        }
      }
    }
    return pipelineIdsToDestinationIndices;
  }

  /**
   * Retrieves the network map and metadata of model ids, pipelines, and indices that are tied to the model ids.
   * @param modelIds - Array of models ids and model aliases.
   */
  async getModelsPipelinesAndIndicesMap(modelId, {
    withIndices
  }) {
    const result = {
      ingestPipelines: new Map(),
      indices: [],
      elements: [],
      details: {},
      error: null
    };
    let pipelinesResponse;
    try {
      var _pipelinesResponse;
      pipelinesResponse = await this.getModelsPipelines([modelId]);

      // 1. Get list of pipelines that are related to the model
      const pipelines = (_pipelinesResponse = pipelinesResponse) === null || _pipelinesResponse === void 0 ? void 0 : _pipelinesResponse.get(modelId);
      const modelNodeId = this.getNodeId(modelId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.TRAINED_MODEL);
      if (pipelines) {
        const pipelineIds = new Set(Object.keys(pipelines));
        result.ingestPipelines = pipelinesResponse;
        for (const pipelineId of pipelineIds) {
          const pipelineNodeId = this.getNodeId(pipelineId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INGEST_PIPELINE);
          result.details[pipelineNodeId] = pipelines[pipelineId];
          result.elements.push({
            data: {
              id: pipelineNodeId,
              label: pipelineId,
              type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INGEST_PIPELINE
            }
          });
          result.elements.push({
            data: {
              id: `${modelNodeId}~${pipelineNodeId}`,
              source: modelNodeId,
              target: pipelineNodeId
            }
          });
        }
        if (withIndices === true) {
          const pipelineIdsToDestinationIndices = await this.getPipelineToIndicesMap(pipelineIds);

          // 3. Grab index information for all the indices found, and add their info to the map
          for (const [pipelineId, indexIds] of Object.entries(pipelineIdsToDestinationIndices)) {
            const pipelineNodeId = this.getNodeId(pipelineId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INGEST_PIPELINE);
            for (const destinationIndexId of indexIds) {
              const destinationIndexNodeId = this.getNodeId(destinationIndexId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX);
              const destinationIndexDetails = await this.getIndexData(destinationIndexId);
              result.indices.push(destinationIndexDetails);
              result.details[destinationIndexNodeId] = {
                ...destinationIndexDetails,
                ml_inference_models: [modelId]
              };
              result.elements.push({
                data: {
                  id: destinationIndexNodeId,
                  label: destinationIndexId,
                  type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX
                }
              });
              result.elements.push({
                data: {
                  id: `${pipelineNodeId}~${destinationIndexNodeId}`,
                  source: pipelineNodeId,
                  target: destinationIndexNodeId
                }
              });
            }
          }
          const destinationIndices = (0, _lodash.flatten)(Object.values(pipelineIdsToDestinationIndices));

          // 4. From these destination indices, check if there's any transforms that have the indexId as the source destination index
          if (destinationIndices.length > 0) {
            const transforms = await this.initTransformData();
            if (!transforms) return result;
            for (const destinationIndex of destinationIndices) {
              const destinationIndexNodeId = `${destinationIndex}-${_mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX}`;
              const foundTransform = transforms === null || transforms === void 0 ? void 0 : transforms.find(t => {
                const transformSourceIndex = Array.isArray(t.source.index) ? t.source.index[0] : t.source.index;
                return transformSourceIndex === destinationIndex;
              });

              // 5. If any of the transforms use these indices as source , find the destination indices to complete the map
              if (foundTransform) {
                const transformDestIndex = foundTransform.dest.index;
                const transformNodeId = `${foundTransform.id}-${_mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.TRANSFORM}`;
                const transformDestIndexNodeId = `${transformDestIndex}-${_mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX}`;
                const destIndex = await this.getIndexData(transformDestIndex);
                result.indices.push(destIndex);
                result.details[transformNodeId] = foundTransform;
                result.details[transformDestIndexNodeId] = destIndex;
                result.elements.push({
                  data: {
                    id: transformNodeId,
                    label: foundTransform.id,
                    type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.TRANSFORM
                  }
                }, {
                  data: {
                    id: transformDestIndexNodeId,
                    label: transformDestIndex,
                    type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX
                  }
                });
                result.elements.push({
                  data: {
                    id: `${destinationIndexNodeId}~${transformNodeId}`,
                    source: destinationIndexNodeId,
                    target: transformNodeId
                  }
                }, {
                  data: {
                    id: `${transformNodeId}~${transformDestIndexNodeId}`,
                    source: transformNodeId,
                    target: transformDestIndexNodeId
                  }
                });
              }
            }
          }
        }
      }
      return result;
    } catch (error) {
      if (error.statusCode === 404) {
        // ES returns 404 when there are no pipelines
        // Instead, we should return the modelIdsMap and a 200
        return result;
      }
      throw error;
    }
  }

  /**
   * Deletes associated pipelines of the requested model
   * @param modelIds
   */
  async deleteModelPipelines(modelIds) {
    const pipelines = await this.getModelsPipelines(modelIds);
    const pipelinesIds = [...new Set([...pipelines.values()].flatMap(v => Object.keys(v)))];
    await Promise.all(pipelinesIds.map(id => this._client.asCurrentUser.ingest.deletePipeline({
      id
    })));
  }

  /**
   * Returns a list of elastic curated models available for download.
   */
  async getModelDownloads() {
    var _this$_cloud;
    // We assume that ML nodes in Cloud are always on linux-x86_64, even if other node types aren't.
    const isCloud = !!((_this$_cloud = this._cloud) !== null && _this$_cloud !== void 0 && _this$_cloud.cloudId);
    const nodesInfoResponse = await this._client.asInternalUser.transport.request({
      method: 'GET',
      path: `/_nodes/ml:true/os`
    });
    let osName;
    let arch;
    // Indicates that all ML nodes have the same architecture
    let sameArch = true;
    for (const node of Object.values(nodesInfoResponse.nodes)) {
      var _node$os3, _node$os4;
      if (!osName) {
        var _node$os;
        osName = (_node$os = node.os) === null || _node$os === void 0 ? void 0 : _node$os.name;
      }
      if (!arch) {
        var _node$os2;
        arch = (_node$os2 = node.os) === null || _node$os2 === void 0 ? void 0 : _node$os2.arch;
      }
      if (((_node$os3 = node.os) === null || _node$os3 === void 0 ? void 0 : _node$os3.name) !== osName || ((_node$os4 = node.os) === null || _node$os4 === void 0 ? void 0 : _node$os4.arch) !== arch) {
        sameArch = false;
        break;
      }
    }
    const modelDefinitionMap = new Map();
    for (const [modelId, def] of Object.entries(_mlTrainedModelsUtils.ELASTIC_MODEL_DEFINITIONS)) {
      const recommended = isCloud && def.os === 'Linux' && def.arch === 'amd64' || sameArch && !!(def !== null && def !== void 0 && def.os) && (def === null || def === void 0 ? void 0 : def.os) === osName && (def === null || def === void 0 ? void 0 : def.arch) === arch;
      const {
        modelName
      } = def;
      const modelDefinitionResponse = {
        ...def,
        ...(recommended ? {
          recommended
        } : {}),
        supported: !!def.default || recommended,
        model_id: modelId
      };
      if (modelDefinitionMap.has(modelName)) {
        modelDefinitionMap.get(modelName).push(modelDefinitionResponse);
      } else {
        modelDefinitionMap.set(modelName, [modelDefinitionResponse]);
      }
    }

    // check if there is no recommended, so we mark default as recommended
    for (const arr of modelDefinitionMap.values()) {
      const defaultModel = arr.find(a => a.default);
      const recommendedModel = arr.find(a => a.recommended);
      if (defaultModel && !recommendedModel) {
        delete defaultModel.default;
        defaultModel.recommended = true;
      }
    }
    return [...modelDefinitionMap.values()].flat();
  }

  /**
   * Provides an appropriate model ID and configuration for download based on the current cluster architecture.
   *
   * @param modelName
   * @param options
   * @returns
   */
  async getCuratedModelConfig(modelName, options) {
    const modelDownloadConfig = (await this.getModelDownloads()).filter(model => model.modelName === modelName);
    let requestedModel;
    let recommendedModel;
    let defaultModel;
    for (const model of modelDownloadConfig) {
      if ((options === null || options === void 0 ? void 0 : options.version) === model.version) {
        requestedModel = model;
        if (model.recommended) {
          requestedModel = model;
          break;
        }
      } else if (model.recommended) {
        recommendedModel = model;
      } else if (model.default) {
        defaultModel = model;
      }
    }
    if (!requestedModel && !defaultModel && !recommendedModel) {
      throw new Error('Requested model not found');
    }
    return requestedModel || recommendedModel || defaultModel;
  }

  /**
   * Provides an ELSER model name and configuration for download based on the current cluster architecture.
   * The current default version is 2. If running on Cloud it returns the Linux x86_64 optimized version.
   * If any of the ML nodes run a different OS rather than Linux, or the CPU architecture isn't x86_64,
   * a portable version of the model is returned.
   */
  async getELSER(options) {
    return await this.getCuratedModelConfig('elser', options);
  }

  /**
   * Puts the requested ELSER model into elasticsearch, triggering elasticsearch to download the model.
   * Assigns the model to the * space.
   * @param modelId
   * @param mlSavedObjectService
   */
  async installElasticModel(modelId, mlSavedObjectService) {
    const availableModels = await this.getModelDownloads();
    const model = availableModels.find(m => m.model_id === modelId);
    if (!model) {
      throw _boom.default.notFound('Model not found');
    }
    let esModelExists = false;
    try {
      await this._client.asInternalUser.ml.getTrainedModels({
        model_id: modelId
      });
      esModelExists = true;
    } catch (error) {
      if (error.statusCode !== 404) {
        throw error;
      }
      // model doesn't exist, ignore error
    }
    if (esModelExists) {
      throw _boom.default.badRequest('Model already exists');
    }
    const putResponse = await this._mlClient.putTrainedModel({
      model_id: model.model_id,
      body: model.config
    });
    await mlSavedObjectService.updateTrainedModelsSpaces([modelId], ['*'], []);
    return putResponse;
  }
  /**
   * Puts the requested Inference endpoint id into elasticsearch, triggering elasticsearch to create the inference endpoint id
   * @param inferenceId - Inference Endpoint Id
   * @param taskType - Inference Task type. Either sparse_embedding or text_embedding
   * @param inferenceConfig - Model configuration based on service type
   */
  async createInferenceEndpoint(inferenceId, taskType, inferenceConfig) {
    try {
      const result = await this._client.asCurrentUser.inference.put({
        inference_id: inferenceId,
        task_type: taskType,
        inference_config: inferenceConfig
      }, {
        maxRetries: 0
      });
      return result;
    } catch (error) {
      // Request timeouts will usually occur when the model is being downloaded/deployed
      // Erroring out is misleading in these cases, so we return the model_id and task_type
      if (error.name === 'TimeoutError') {
        return {
          model_id: inferenceConfig.service,
          task_type: taskType
        };
      } else {
        throw error;
      }
    }
  }
  async getModelsDownloadStatus() {
    var _result$tasks;
    const result = await this._client.asInternalUser.tasks.list({
      actions: 'xpack/ml/model_import[n]',
      detailed: true,
      group_by: 'none'
    });
    if (!((_result$tasks = result.tasks) !== null && _result$tasks !== void 0 && _result$tasks.length)) {
      return {};
    }

    // Groups results by model id
    const byModelId = result.tasks.reduce((acc, task) => {
      const modelId = task.description.replace(`model_id-`, '');
      acc[modelId] = {
        downloaded_parts: task.status.downloaded_parts,
        total_parts: task.status.total_parts
      };
      return acc;
    }, {});
    return byModelId;
  }
}
exports.ModelsProvider = ModelsProvider;