"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MapsStorage = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _common = require("../../common");
var _transform_utils = require("./schema/v1/transform_utils");
var _cm_services = require("./schema/cm_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const savedObjectClientFromRequest = async ctx => {
  if (!ctx.requestHandlerContext) {
    throw new Error('Storage context.requestHandlerContext missing.');
  }
  const {
    savedObjects
  } = await ctx.requestHandlerContext.core;
  return savedObjects.client;
};
const searchArgsToSOFindOptions = (query, options) => {
  var _query$tags, _query$tags$included, _query$tags2, _query$tags2$excluded;
  const hasReference = (_query$tags = query.tags) === null || _query$tags === void 0 ? void 0 : (_query$tags$included = _query$tags.included) === null || _query$tags$included === void 0 ? void 0 : _query$tags$included.map(tagId => ({
    type: 'tag',
    id: tagId
  }));
  const hasNoReference = (_query$tags2 = query.tags) === null || _query$tags2 === void 0 ? void 0 : (_query$tags2$excluded = _query$tags2.excluded) === null || _query$tags2$excluded === void 0 ? void 0 : _query$tags2$excluded.map(tagId => ({
    type: 'tag',
    id: tagId
  }));
  return {
    type: _common.MAP_SAVED_OBJECT_TYPE,
    searchFields: options !== null && options !== void 0 && options.onlyTitle ? ['title'] : ['title^3', 'description'],
    search: query.text,
    perPage: query.limit,
    page: query.cursor ? +query.cursor : undefined,
    defaultSearchOperator: 'AND',
    hasReference,
    hasNoReference
  };
};
class MapsStorage {
  constructor({
    logger,
    throwOnResultValidationError
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "throwOnResultValidationError", void 0);
    (0, _defineProperty2.default)(this, "mSearch", {
      savedObjectType: _common.MAP_SAVED_OBJECT_TYPE,
      toItemResult: (ctx, savedObject) => {
        const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
        const contentItem = (0, _transform_utils.savedObjectToItem)(savedObject, false);
        const validationError = transforms.mSearch.out.result.validate(contentItem);
        if (validationError) {
          if (this.throwOnResultValidationError) {
            throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
          } else {
            this.logger.warn(`Invalid response. ${validationError.message}`);
          }
        }

        // Validate DB response and DOWN transform to the request version
        const {
          value,
          error: resultError
        } = transforms.mSearch.out.result.down(contentItem, undefined,
        // do not override version
        {
          validate: false
        } // validation is done above
        );
        if (resultError) {
          throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
        }
        return value;
      }
    });
    this.logger = logger;
    this.throwOnResultValidationError = throwOnResultValidationError !== null && throwOnResultValidationError !== void 0 ? throwOnResultValidationError : false;
  }
  async get(ctx, id) {
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);
    const {
      saved_object: savedObject,
      alias_purpose: aliasPurpose,
      alias_target_id: aliasTargetId,
      outcome
    } = await soClient.resolve(_common.MAP_SAVED_OBJECT_TYPE, id);
    const response = {
      item: savedObject,
      meta: {
        aliasPurpose,
        aliasTargetId,
        outcome
      }
    };
    const validationError = transforms.get.out.result.validate(response);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }
    const {
      value,
      error: resultError
    } = transforms.get.out.result.down(response, undefined, {
      validate: false
    });
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async bulkGet() {
    // Not implemented
    throw new Error(`[bulkGet] has not been implemented. See MapsStorage class.`);
  }
  async create(ctx, data, options) {
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);

    // Validate input (data & options) & UP transform them to the latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.create.in.data.up(data);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.create.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const {
      attributes: soAttributes,
      references: soReferences
    } = (0, _transform_utils.itemToSavedObject)({
      attributes: dataToLatest,
      references: options.references
    });

    // Save data in DB
    const savedObject = await soClient.create(_common.MAP_SAVED_OBJECT_TYPE, soAttributes, {
      ...optionsToLatest,
      references: soReferences
    });
    const item = (0, _transform_utils.savedObjectToItem)(savedObject, false);
    const validationError = transforms.create.out.result.validate({
      item
    });
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate DB response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.create.out.result.down({
      item
    }, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async update(ctx, id, data, options) {
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);

    // Validate input (data & options) & UP transform them to the latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.update.in.data.up(data);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.update.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const {
      attributes: soAttributes,
      references: soReferences
    } = (0, _transform_utils.itemToSavedObject)({
      attributes: dataToLatest,
      references: options.references
    });

    // Save data in DB
    const partialSavedObject = await soClient.update(_common.MAP_SAVED_OBJECT_TYPE, id, soAttributes, {
      ...optionsToLatest,
      references: soReferences
    });
    const item = (0, _transform_utils.savedObjectToItem)(partialSavedObject, true);
    const validationError = transforms.update.out.result.validate({
      item
    });
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate DB response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.update.out.result.down({
      item
    }, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async delete(ctx, id,
  // force is necessary to delete saved objects that exist in multiple namespaces
  options) {
    var _options$force;
    const soClient = await savedObjectClientFromRequest(ctx);
    await soClient.delete(_common.MAP_SAVED_OBJECT_TYPE, id, {
      force: (_options$force = options === null || options === void 0 ? void 0 : options.force) !== null && _options$force !== void 0 ? _options$force : false
    });
    return {
      success: true
    };
  }
  async search(ctx, query, options) {
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.search.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid payload. ${optionsError.message}`);
    }
    const soQuery = searchArgsToSOFindOptions(query, optionsToLatest);
    const soResponse = await soClient.find(soQuery);
    const hits = await Promise.all(soResponse.saved_objects.map(async so => {
      const item = (0, _transform_utils.savedObjectToItem)(so, false);
      return item;
    }).filter(item => item !== null));
    const response = {
      hits,
      pagination: {
        total: soResponse.total
      }
    };
    const validationError = transforms.search.out.result.validate(response);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }
    const {
      value,
      error: resultError
    } = transforms.search.out.result.down(response, undefined, {
      validate: false
    });
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
}
exports.MapsStorage = MapsStorage;